import os
import pandas as pd
import csv
import re
from bs4 import BeautifulSoup
import chardet
from openpyxl import load_workbook
from openpyxl.chart import LineChart, Reference
from openpyxl.utils.dataframe import dataframe_to_rows

def detect_encoding(file_path):
    """Определяет кодировку файла"""
    with open(file_path, 'rb') as f:
        raw_data = f.read()
        result = chardet.detect(raw_data)
        return result['encoding']

def normalize_key(key):
    """Нормализует имя поля для сравнения"""
    if not key:
        return ''
    key = key.strip().lower()
    key = re.sub(r'[:\s]+', ' ', key)
    key = re.sub(r'\s*\(\s*', ' (', key)
    key = re.sub(r'\s+\)', ')', key)
    return key

def parse_stat_value(value, field_name=''):
    """Парсит значение, пытаясь преобразовать в число, если это возможно"""
    if not value:
        return value
    
    integer_fields = [
        'непрерывный выигрыш',
        'непрерывный проигрыш',
        'непрерывных выигрышей',
        'непрерывных проигрышей',
        'всего сделок',
        'короткие позиции',
        'длинные позиции',
        'прибыльные сделки',
        'убыточные сделки'
    ]
    
    is_integer_field = any(x in field_name.lower() for x in integer_fields)
    
    if re.search(r'[а-яА-Яa-zA-Z%\(\)]', value):
        return value.strip()
    
    try:
        cleaned = value.strip().replace(',', '.').replace(' ', '')
        if cleaned and cleaned != '-':
            num_val = float(cleaned)
            if is_integer_field and num_val == int(num_val):
                return int(num_val)
            return num_val
    except:
        pass
    
    return value.strip()

def add_balance_chart(writer, trades_data, trades_headers):
    """Добавляет график изменения баланса в Excel"""
    if not trades_data or 'Баланс' not in trades_headers:
        return
    
    workbook = writer.book
    chart_sheet = workbook.create_sheet('График баланса')
    
    balance_idx = trades_headers.index('Баланс')
    balances = []
    
    for i, trade in enumerate(trades_data):
        if balance_idx < len(trade) and trade[balance_idx]:
            try:
                balance = float(trade[balance_idx]) if not isinstance(trade[balance_idx], (int, float)) else trade[balance_idx]
                balances.append([i+1, balance])
            except:
                pass
    
    if not balances:
        return
    
    df_balance = pd.DataFrame(balances, columns=['Сделка №', 'Баланс'])
    
    for r in dataframe_to_rows(df_balance, index=False, header=True):
        chart_sheet.append(r)
    
    chart = LineChart()
    chart.title = "Изменение баланса в ходе торговли"
    chart.style = 10
    chart.x_axis.title = "Номер сделки"
    chart.y_axis.title = "Баланс"
    chart.height = 15
    chart.width = 25
    
    data = Reference(chart_sheet, min_col=2, min_row=1, max_row=len(df_balance)+1)
    categories = Reference(chart_sheet, min_col=1, min_row=2, max_row=len(df_balance)+1)
    chart.add_data(data, titles_from_data=True)
    chart.set_categories(categories)
    
    chart_sheet.add_chart(chart, "D2")

def save_to_csv(csv_filename, summary_data, trades_data, trades_headers):
    """Сохраняет данные в CSV-файл с разделителями"""
    with open(csv_filename, 'w', newline='', encoding='utf-8-sig') as csvfile:
        writer = csv.writer(csvfile, delimiter=';')
        
        writer.writerow(['=== ОТЧЕТ ТЕСТЕРА СТРАТЕГИЙ ==='])
        writer.writerow([])
        
        stats_dict = {item['Показатель']: item['Значение'] for item in summary_data}
        
        writer.writerow(['ОСНОВНАЯ ИНФОРМАЦИЯ'])
        main_info_keys = ['Символ', 'Период', 'Модель', 'Параметры', 'Баров в истории', 
                         'Смоделировано тиков', 'Качество моделирования', 'Ошибки рассогласования графиков']
        
        for key in main_info_keys:
            if key in stats_dict:
                writer.writerow([key, stats_dict[key]])
        writer.writerow([])
        
        writer.writerow(['ИТОГОВАЯ СТАТИСТИКА'])
        writer.writerow(['Показатель', 'Значение'])
        
        base_keys = ['Начальный депозит', 'Чистая прибыль', 'Общая прибыль', 'Общий убыток',
                    'Прибыльность', 'Матожидание выигрыша', 'Абсолютная просадка', 
                    'Максимальная просадка', 'Относительная просадка', 'Спред']
        
        for key in base_keys:
            if key in stats_dict:
                writer.writerow([key, stats_dict[key]])
        
        writer.writerow([])
        
        writer.writerow(['РАСШИРЕННАЯ СТАТИСТИКА СДЕЛОК'])
        writer.writerow(['Показатель', 'Значение'])
        
        extended_fields = [
            'Всего сделок',
            'Короткие позиции (% выигравших)',
            'Длинные позиции (% выигравших)',
            'Прибыльные сделки (% от всех)',
            'Убыточные сделки (% от всех)',
            'Самая большая прибыльная сделка',
            'Самая большая убыточная сделка',
            'Средняя прибыльная сделка',
            'Средняя убыточная сделка',
            'Максимальное количество непрерывных выигрышей (прибыль)',
            'Максимальное количество непрерывных проигрышей (убыток)',
            'Макс. непрерывная прибыль (число выигрышей)',
            'Макс. непрерывный убыток (число проигрышей)',
            'Средний непрерывный выигрыш',
            'Средний непрерывный проигрыш'
        ]
        
        for field in extended_fields:
            found = False
            field_norm = normalize_key(field)
            
            if field in stats_dict:
                writer.writerow([field, stats_dict[field]])
                found = True
            else:
                for key, value in stats_dict.items():
                    key_norm = normalize_key(key)
                    if field_norm == key_norm:
                        writer.writerow([field, value])
                        found = True
                        break
            
            if not found:
                writer.writerow([field, ''])
        
        writer.writerow([])
        writer.writerow([])
        
        if trades_data and trades_headers:
            writer.writerow(['СПИСОК СДЕЛОК'])
            writer.writerow(trades_headers)
            for trade in trades_data:
                writer.writerow(trade)

def parse_mt4_to_files(html_file_path, output_excel="mt4_report.xlsx", output_csv="mt4_report.csv"):
    """
    Парсит HTML-отчет тестера стратегий MT4 и сохраняет:
    - Excel с тремя листами и графиком
    - CSV с полными данными
    """
    
    if not os.path.exists(html_file_path):
        print(f"Ошибка: Файл {html_file_path} не найден.")
        return

    encoding = detect_encoding(html_file_path)
    
    try:
        with open(html_file_path, 'r', encoding=encoding) as f:
            html_content = f.read()
    except UnicodeDecodeError:
        for enc in ['utf-8', 'windows-1251', 'cp1251']:
            try:
                with open(html_file_path, 'r', encoding=enc) as f:
                    html_content = f.read()
                encoding = enc
                break
            except UnicodeDecodeError:
                continue

    soup = BeautifulSoup(html_content, 'html.parser')
    
    summary_data = []
    tables = soup.find_all('table')
    
    for tbl_idx, table in enumerate(tables):
        rows = table.find_all('tr')
        
        for row in rows:
            cols = row.find_all('td')
            
            if not cols or (len(cols) == 1 and not cols[0].get_text(strip=True)):
                continue
            
            if len(cols) == 5:
                prefix = cols[0].get_text(strip=True)
                key1 = (prefix + " " + cols[1].get_text(strip=True)).strip() if prefix else cols[1].get_text(strip=True)
                value1 = cols[2].get_text(strip=True)
                key2 = (prefix + " " + cols[3].get_text(strip=True)).strip() if prefix else cols[3].get_text(strip=True)
                value2 = cols[4].get_text(strip=True)
                
                if key1 and value1:
                    summary_data.append({
                        'Показатель': key1,
                        'Значение': parse_stat_value(value1, key1)
                    })
                if key2 and value2:
                    summary_data.append({
                        'Показатель': key2,
                        'Значение': parse_stat_value(value2, key2)
                    })
                continue
            
            if len(cols) == 2:
                key = cols[0].get_text(strip=True)
                value = cols[1].get_text(strip=True)
                if key and value:
                    summary_data.append({
                        'Показатель': key,
                        'Значение': parse_stat_value(value, key)
                    })
            
            elif len(cols) == 4:
                for i in range(0, 4, 2):
                    key = cols[i].get_text(strip=True)
                    if i+1 < len(cols):
                        value = cols[i+1].get_text(strip=True)
                        if key and value:
                            summary_data.append({
                                'Показатель': key,
                                'Значение': parse_stat_value(value, key)
                            })
            
            elif len(cols) == 6:
                for i in range(0, 6, 2):
                    key = cols[i].get_text(strip=True)
                    if i+1 < len(cols):
                        value = cols[i+1].get_text(strip=True)
                        if key and value:
                            summary_data.append({
                                'Показатель': key,
                                'Значение': parse_stat_value(value, key)
                            })
    
    trades_data = []
    trades_headers = []
    
    if len(tables) > 1:
        trades_table = tables[1]
        rows = trades_table.find_all('tr')
        
        if rows:
            trades_headers = [td.get_text(strip=True) for td in rows[0].find_all('td')]
            
            order_idx = None
            if 'Ордер' in trades_headers:
                order_idx = trades_headers.index('Ордер')
            
            for row in rows[1:]:
                cols = row.find_all('td')
                if cols:
                    trade = []
                    for i, col in enumerate(cols):
                        value = col.get_text(strip=True)
                        
                        if i == order_idx:
                            try:
                                trade.append(int(re.sub(r'[^\d]', '', value)))
                            except:
                                trade.append(value)
                        elif i >= 4:
                            try:
                                cleaned = re.sub(r'[^\d.-]', '', value.replace(',', '.'))
                                if cleaned and cleaned != '-':
                                    trade.append(float(cleaned))
                                else:
                                    trade.append(None)
                            except:
                                trade.append(value)
                        else:
                            trade.append(value)
                    trades_data.append(trade)
    
    with pd.ExcelWriter(output_excel, engine='openpyxl') as writer:
        if summary_data:
            df_summary = pd.DataFrame(summary_data)
            df_summary.to_excel(writer, sheet_name='Полная статистика', index=False)
        
        if trades_data and trades_headers:
            df_trades = pd.DataFrame(trades_data, columns=trades_headers)
            df_trades.to_excel(writer, sheet_name='Сделки', index=False)
        
        if summary_data and trades_data:
            metrics = {}
            for item in summary_data:
                key = item['Показатель']
                if any(x in key for x in ['Чистая прибыль', 'Всего сделок', 'Прибыльность', 
                                         'Максимальная просадка', 'Начальный депозит',
                                         'Прибыльные сделки', 'Убыточные сделки']):
                    try:
                        val = str(item['Значение'])
                        num_match = re.search(r'-?\d+\.?\d*', val)
                        if num_match:
                            metrics[key] = float(num_match.group())
                        else:
                            metrics[key] = val
                    except:
                        metrics[key] = item['Значение']
            
            if metrics:
                df_metrics = pd.DataFrame(list(metrics.items()), columns=['Метрика', 'Значение'])
                df_metrics.to_excel(writer, sheet_name='Аналитика', index=False)
        
        if trades_data and trades_headers:
            add_balance_chart(writer, trades_data, trades_headers)
    
    save_to_csv(output_csv, summary_data, trades_data, trades_headers)
    
    print(f"\n✅ ГОТОВО! Созданы файлы:")
    print(f"   📊 Excel: {output_excel}")
    print(f"   📄 CSV: {output_csv}")

if __name__ == "__main__":
    filename = "StrategyTester.htm"
    parse_mt4_to_files(filename)