//+------------------------------------------------------------------+
//|                                               TickCompressor.mqh |
//|                                  Simplified Balanced Compressor  |
//|                     Based on Python prototype testing results    |
//+------------------------------------------------------------------+
#property copyright "Based on forum discussion and Python testing"
#property link      "https://www.mql5.com/ru/forum/499639"
#property version   "1.00"

//+------------------------------------------------------------------+
//| Константы компрессора                                            |
//+------------------------------------------------------------------+
#define LUT_SIZE 512
#define LUT_HALF 256
#define FAST_PATH_THRESHOLD 8        // дельты от -8 до +8 (72% тиков)
#define FLAG_FAST_PATH 0x00          // старший бит = 0
#define FLAG_SLOW_PATH 0x80          // старший бит = 1

//+------------------------------------------------------------------+
//| Статистика компрессии                                            |
//+------------------------------------------------------------------+
struct CompressionStats
{
   int fast_path_count;              // количество тиков в fast path
   int slow_path_count;              // количество тиков в slow path
   int total_ticks;                  // всего тиков
   double compression_ratio;         // степень сжатия
};

//+------------------------------------------------------------------+
//| Класс компрессора тиков                                          |
//+------------------------------------------------------------------+
class CTickCompressor
{
private:
   double            m_delta_lut[];           // Lookup Table для дельт
   double            m_point;                 // размер пункта
   int               m_digits;                // количество знаков
   CompressionStats  m_stats;                 // статистика
   
   // Вспомогательные функции
   void              InitLUT();
   int               GetDeltaSize(int delta);
   void              WriteDelta(uchar &buffer[], int &pos, int delta, int size);
   int               ReadDelta(const uchar &buffer[], int &pos, int size);
   
public:
                     CTickCompressor();
                    ~CTickCompressor();
   
   // Основные методы
   int               Compress(const MqlTick &ticks[], uchar &compressed[], int count);
   int               Decompress(const uchar &compressed[], MqlTick &ticks[], int compressed_size, int max_ticks=-1);
   
   // Получение статистики
   void              GetStats(CompressionStats &stats) { stats = m_stats; }
   void              ResetStats();
   void              PrintStats();
};

//+------------------------------------------------------------------+
//| Конструктор                                                       |
//+------------------------------------------------------------------+
CTickCompressor::CTickCompressor()
{
   m_point = _Point;
   m_digits = _Digits;
   ArrayResize(m_delta_lut, LUT_SIZE);
   InitLUT();
   ResetStats();
}

//+------------------------------------------------------------------+
//| Деструктор                                                        |
//+------------------------------------------------------------------+
CTickCompressor::~CTickCompressor()
{
   ArrayFree(m_delta_lut);
}

//+------------------------------------------------------------------+
//| Инициализация Lookup Table                                       |
//| КРИТИЧНО: Предвычисляем delta * _Point для быстрой декомпрессии |
//+------------------------------------------------------------------+
void CTickCompressor::InitLUT()
{
   for(int i = 0; i < LUT_SIZE; i++)
   {
      int delta = i - LUT_HALF;
      m_delta_lut[i] = delta * m_point;
   }
}

//+------------------------------------------------------------------+
//| Сброс статистики                                                  |
//+------------------------------------------------------------------+
void CTickCompressor::ResetStats()
{
   m_stats.fast_path_count = 0;
   m_stats.slow_path_count = 0;
   m_stats.total_ticks = 0;
   m_stats.compression_ratio = 0.0;
}

//+------------------------------------------------------------------+
//| Вывод статистики                                                  |
//+------------------------------------------------------------------+
void CTickCompressor::PrintStats()
{
   if(m_stats.total_ticks == 0) return;
   
   Print("=== Compression Statistics ===");
   Print("Total ticks: ", m_stats.total_ticks);
   Print("Fast path: ", m_stats.fast_path_count, " (", 
         (double)m_stats.fast_path_count/m_stats.total_ticks*100, "%)");
   Print("Slow path: ", m_stats.slow_path_count, " (", 
         (double)m_stats.slow_path_count/m_stats.total_ticks*100, "%)");
   Print("Compression ratio: ", DoubleToString(m_stats.compression_ratio, 2), "x");
}

//+------------------------------------------------------------------+
//| Определить размер дельты в байтах                                |
//+------------------------------------------------------------------+
int CTickCompressor::GetDeltaSize(int delta)
{
   if(delta >= -128 && delta <= 127)
      return 0;  // 1 байт
   else if(delta >= -32768 && delta <= 32767)
      return 1;  // 2 байта
   else if(delta >= -2147483648 && delta <= 2147483647)
      return 2;  // 4 байта
   else
      return 3;  // 8 байтов
}

//+------------------------------------------------------------------+
//| Записать дельту переменной длины                                 |
//+------------------------------------------------------------------+
void CTickCompressor::WriteDelta(uchar &buffer[], int &pos, int delta, int size)
{
   switch(size)
   {
      case 0:  // 1 байт (signed char)
         buffer[pos++] = (uchar)(delta & 0xFF);
         break;
         
      case 1:  // 2 байта (short)
         buffer[pos++] = (uchar)(delta & 0xFF);
         buffer[pos++] = (uchar)((delta >> 8) & 0xFF);
         break;
         
      case 2:  // 4 байта (int)
         buffer[pos++] = (uchar)(delta & 0xFF);
         buffer[pos++] = (uchar)((delta >> 8) & 0xFF);
         buffer[pos++] = (uchar)((delta >> 16) & 0xFF);
         buffer[pos++] = (uchar)((delta >> 24) & 0xFF);
         break;
         
      case 3:  // 8 байтов (long)
         long long_delta = (long)delta;
         for(int i = 0; i < 8; i++)
            buffer[pos++] = (uchar)((long_delta >> (i*8)) & 0xFF);
         break;
   }
}

//+------------------------------------------------------------------+
//| Прочитать дельту переменной длины                                |
//+------------------------------------------------------------------+
int CTickCompressor::ReadDelta(const uchar &buffer[], int &pos, int size)
{
   int delta = 0;
   
   switch(size)
   {
      case 0:  // 1 байт (signed char)
         delta = (char)buffer[pos++];
         break;
         
      case 1:  // 2 байта (signed short)
         delta = (short)(buffer[pos] | (buffer[pos+1] << 8));
         pos += 2;
         break;
         
      case 2:  // 4 байта (signed int)
         delta = (int)(buffer[pos] | (buffer[pos+1] << 8) | 
                      (buffer[pos+2] << 16) | (buffer[pos+3] << 24));
         pos += 4;
         break;
         
      case 3:  // 8 байтов (signed long)
         long long_delta = 0;
         for(int i = 0; i < 8; i++)
            long_delta |= ((long)buffer[pos++] << (i*8));
         delta = (int)long_delta;
         break;
   }
   
   return delta;
}

//+------------------------------------------------------------------+
//| Сжать массив тиков                                               |
//| ОПТИМИЗАЦИЯ: Предвыделенный большой массив, один ArrayResize     |
//+------------------------------------------------------------------+
int CTickCompressor::Compress(const MqlTick &ticks[], uchar &compressed[], int count)
{
   if(count <= 0) return 0;
   
   // Предварительно выделить память (максимум 10 байт на тик)
   int max_size = 24 + count * 10;  // заголовок + данные
   ArrayResize(compressed, max_size);
   
   int pos = 0;
   
   // 1. Записать заголовок - первый тик полностью
   long first_time = (long)ticks[0].time;
   for(int i = 0; i < 8; i++)
      compressed[pos++] = (uchar)((first_time >> (i*8)) & 0xFF);
   
   // Записать bid и ask как double (8 байт каждый)
   uchar bid_bytes[], ask_bytes[];
   ArrayResize(bid_bytes, 8);
   ArrayResize(ask_bytes, 8);
   
   // Копировать байты double в массив
   double first_bid = ticks[0].bid;
   double first_ask = ticks[0].ask;
   
   for(int i = 0; i < 8; i++)
   {
      compressed[pos++] = ((uchar*)&first_bid)[i];
      compressed[pos++] = ((uchar*)&first_ask)[i];
   }
   
   // 2. Сжать остальные тики
   double last_bid = first_bid;
   double last_ask = first_ask;
   long last_time = first_time;
   
   int fast_count = 0;
   int slow_count = 0;
   
   for(int i = 1; i < count; i++)
   {
      // Вычислить дельты
      int bid_delta = (int)MathRound((ticks[i].bid - last_bid) / m_point);
      int ask_delta = (int)MathRound((ticks[i].ask - last_ask) / m_point);
      int time_delta = (int)(ticks[i].time - last_time);
      
      // Fast path: малые дельты (72% случаев!)
      if(MathAbs(bid_delta) <= FAST_PATH_THRESHOLD && 
         MathAbs(ask_delta) <= FAST_PATH_THRESHOLD &&
         time_delta >= 0 && time_delta <= 255)
      {
         compressed[pos++] = FLAG_FAST_PATH;
         compressed[pos++] = (uchar)(bid_delta & 0xFF);  // signed char
         compressed[pos++] = (uchar)(ask_delta & 0xFF);  // signed char
         compressed[pos++] = (uchar)time_delta;
         
         fast_count++;
      }
      // Slow path: большие дельты
      else
      {
         int bid_size = GetDeltaSize(bid_delta);
         int ask_size = GetDeltaSize(ask_delta);
         int time_size = GetDeltaSize(time_delta);
         
         uchar flags = FLAG_SLOW_PATH;
         flags |= (uchar)((bid_size << 4) | (ask_size << 2) | time_size);
         
         compressed[pos++] = flags;
         WriteDelta(compressed, pos, bid_delta, bid_size);
         WriteDelta(compressed, pos, ask_delta, ask_size);
         WriteDelta(compressed, pos, time_delta, time_size);
         
         slow_count++;
      }
      
      // Обновить последние значения
      last_bid = ticks[i].bid;
      last_ask = ticks[i].ask;
      last_time = (long)ticks[i].time;
   }
   
   // Урезать массив до фактического размера (один раз!)
   ArrayResize(compressed, pos);
   
   // Обновить статистику
   m_stats.fast_path_count = fast_count;
   m_stats.slow_path_count = slow_count;
   m_stats.total_ticks = count;
   
   int original_size = count * (8 + 8 + 8);  // time + bid + ask
   m_stats.compression_ratio = (double)original_size / pos;
   
   return pos;
}

//+------------------------------------------------------------------+
//| Распаковать сжатые данные                                        |
//| ОПТИМИЗАЦИЯ: LUT для 72% тиков, без ArrayResize                 |
//+------------------------------------------------------------------+
int CTickCompressor::Decompress(const uchar &compressed[], MqlTick &ticks[], 
                                 int compressed_size, int max_ticks=-1)
{
   if(compressed_size < 24) return 0;  // минимальный размер заголовка
   
   int pos = 0;
   
   // 1. Прочитать заголовок
   long first_time = 0;
   for(int i = 0; i < 8; i++)
      first_time |= ((long)compressed[pos++] << (i*8));
   
   double first_bid = 0, first_ask = 0;
   for(int i = 0; i < 8; i++)
      ((uchar*)&first_bid)[i] = compressed[pos++];
   for(int i = 0; i < 8; i++)
      ((uchar*)&first_ask)[i] = compressed[pos++];
   
   // Первый тик
   ticks[0].time = (datetime)first_time;
   ticks[0].bid = first_bid;
   ticks[0].ask = first_ask;
   
   double last_bid = first_bid;
   double last_ask = first_ask;
   long last_time = first_time;
   
   int tick_count = 1;
   
   // 2. Декомпрессия остальных тиков
   while(pos < compressed_size && (max_ticks < 0 || tick_count < max_ticks))
   {
      uchar flags = compressed[pos++];
      
      // Fast path - КЛЮЧЕВАЯ ОПТИМИЗАЦИЯ (72% тиков)
      if((flags & FLAG_SLOW_PATH) == 0)
      {
         // Читаем как signed char
         char bid_delta = (char)compressed[pos++];
         char ask_delta = (char)compressed[pos++];
         uchar time_delta = compressed[pos++];
         
         // ИСПОЛЬЗУЕМ LUT - БЕЗ УМНОЖЕНИЯ!
         ticks[tick_count].bid = NormalizeDouble(last_bid + m_delta_lut[bid_delta + LUT_HALF], m_digits);
         ticks[tick_count].ask = NormalizeDouble(last_ask + m_delta_lut[ask_delta + LUT_HALF], m_digits);
         ticks[tick_count].time = (datetime)(last_time + time_delta);
         
         last_bid = ticks[tick_count].bid;
         last_ask = ticks[tick_count].ask;
         last_time = (long)ticks[tick_count].time;
         
         tick_count++;
         continue;
      }
      
      // Slow path - для больших дельт (27% тиков)
      int bid_size = (flags >> 4) & 0x03;
      int ask_size = (flags >> 2) & 0x03;
      int time_size = flags & 0x03;
      
      int bid_delta = ReadDelta(compressed, pos, bid_size);
      int ask_delta = ReadDelta(compressed, pos, ask_size);
      int time_delta = ReadDelta(compressed, pos, time_size);
      
      ticks[tick_count].bid = NormalizeDouble(last_bid + bid_delta * m_point, m_digits);
      ticks[tick_count].ask = NormalizeDouble(last_ask + ask_delta * m_point, m_digits);
      ticks[tick_count].time = (datetime)(last_time + time_delta);
      
      last_bid = ticks[tick_count].bid;
      last_ask = ticks[tick_count].ask;
      last_time = (long)ticks[tick_count].time;
      
      tick_count++;
   }
   
   return tick_count;
}
//+------------------------------------------------------------------+
