//+------------------------------------------------------------------+
//|                                          TestCompressor.mq5      |
//|                              Пример использования компрессора    |
//+------------------------------------------------------------------+
#property copyright "Test Script"
#property version   "1.00"
#property script_show_inputs

#include "TickCompressor.mqh"

//--- Параметры
input int      InpTickCount = 10000;     // Количество тиков для теста
input datetime InpDateFrom = D'2024.01.01'; // Начальная дата
input datetime InpDateTo = D'2024.01.31';   // Конечная дата

//+------------------------------------------------------------------+
//| Script program start function                                    |
//+------------------------------------------------------------------+
void OnStart()
{
   Print("=== Test Tick Compressor ===");
   Print("Symbol: ", _Symbol);
   Print("Point: ", _Point);
   Print("Digits: ", _Digits);
   
   // 1. Загрузить реальные тики
   MqlTick ticks[];
   int tick_count = LoadTicks(ticks, InpDateFrom, InpDateTo, InpTickCount);
   
   if(tick_count <= 0)
   {
      Print("ERROR: No ticks loaded!");
      return;
   }
   
   Print("\n--- Loaded ", tick_count, " ticks ---");
   Print("Time range: ", ticks[0].time, " to ", ticks[tick_count-1].time);
   Print("Price range: ", ticks[0].bid, " to ", ticks[tick_count-1].bid);
   
   // 2. Создать компрессор
   CTickCompressor compressor;
   
   // 3. Сжать данные
   uchar compressed[];
   uint start_time = GetTickCount();
   
   int compressed_size = compressor.Compress(ticks, compressed, tick_count);
   
   uint compress_time = GetTickCount() - start_time;
   
   if(compressed_size <= 0)
   {
      Print("ERROR: Compression failed!");
      return;
   }
   
   // 4. Вывести статистику сжатия
   int original_size = tick_count * 24;  // время(8) + bid(8) + ask(8)
   double ratio = (double)original_size / compressed_size;
   
   Print("\n--- Compression Results ---");
   Print("Original size: ", original_size, " bytes (", original_size/1024.0, " KB)");
   Print("Compressed size: ", compressed_size, " bytes (", compressed_size/1024.0, " KB)");
   Print("Compression ratio: ", DoubleToString(ratio, 2), "x");
   Print("Compression time: ", compress_time, " ms");
   Print("Speed: ", DoubleToString(tick_count/(compress_time/1000.0)/1000000.0, 2), " million ticks/sec");
   
   compressor.PrintStats();
   
   // 5. Распаковать данные
   MqlTick decompressed[];
   ArrayResize(decompressed, tick_count);
   
   start_time = GetTickCount();
   
   int decompressed_count = compressor.Decompress(compressed, decompressed, compressed_size);
   
   uint decompress_time = GetTickCount() - start_time;
   
   Print("\n--- Decompression Results ---");
   Print("Decompressed ticks: ", decompressed_count);
   Print("Decompression time: ", decompress_time, " ms");
   Print("Speed: ", DoubleToString(decompressed_count/(decompress_time/1000.0)/1000000.0, 2), " million ticks/sec");
   
   // 6. Проверить корректность
   int errors = CheckCorrectness(ticks, decompressed, tick_count);
   
   Print("\n--- Verification ---");
   if(errors == 0)
   {
      Print("SUCCESS: All ticks restored correctly!");
   }
   else
   {
      Print("WARNING: Found ", errors, " errors in ", tick_count, " ticks");
   }
   
   // 7. Сравнить с наивным хранением
   int naive_size = tick_count * 24;  // только time+bid+ask
   Print("\n--- Comparison ---");
   Print("Naive storage: ", naive_size, " bytes");
   Print("Our compression: ", compressed_size, " bytes");
   Print("Savings: ", DoubleToString((double)(naive_size - compressed_size)/naive_size*100, 1), "%");
   
   Print("\n=== Test Complete ===");
}

//+------------------------------------------------------------------+
//| Загрузить тики из истории                                        |
//+------------------------------------------------------------------+
int LoadTicks(MqlTick &ticks[], datetime from, datetime to, int max_count)
{
   // Попробовать загрузить тики из терминала
   int loaded = CopyTicks(_Symbol, ticks, COPY_TICKS_ALL, 
                          (ulong)from * 1000, max_count);
   
   if(loaded > 0)
   {
      Print("Loaded ", loaded, " real ticks from terminal");
      return loaded;
   }
   
   // Если не удалось загрузить, генерировать синтетические
   Print("Generating ", max_count, " synthetic ticks...");
   return GenerateSyntheticTicks(ticks, max_count);
}

//+------------------------------------------------------------------+
//| Генерировать синтетические тики для теста                       |
//+------------------------------------------------------------------+
int GenerateSyntheticTicks(MqlTick &ticks[], int count)
{
   ArrayResize(ticks, count);
   
   double base_price = SymbolInfoDouble(_Symbol, SYMBOL_BID);
   double spread = SymbolInfoDouble(_Symbol, SYMBOL_ASK) - base_price;
   
   // Первый тик
   ticks[0].time = TimeCurrent();
   ticks[0].bid = base_price;
   ticks[0].ask = base_price + spread;
   ticks[0].last = 0;
   ticks[0].volume = 0;
   ticks[0].flags = 0;
   
   MathSrand((int)GetTickCount());
   
   double current_price = base_price;
   datetime current_time = ticks[0].time;
   
   // Генерировать тики с небольшими случайными изменениями
   for(int i = 1; i < count; i++)
   {
      // Изменение цены: обычно малое (для Fast Path)
      int price_change = 0;
      int rand_val = MathRand() % 100;
      
      if(rand_val < 60)  // 60% - малые изменения (-2..+2 пункта)
         price_change = (MathRand() % 5) - 2;
      else if(rand_val < 90)  // 30% - средние изменения (-8..+8 пунктов)
         price_change = (MathRand() % 17) - 8;
      else  // 10% - большие изменения
         price_change = (MathRand() % 100) - 50;
      
      current_price += price_change * _Point;
      
      // Время: обычно 10-1000 мс между тиками
      int time_delta = 0;
      rand_val = MathRand() % 100;
      
      if(rand_val < 30)
         time_delta = 10;
      else if(rand_val < 60)
         time_delta = 50;
      else if(rand_val < 80)
         time_delta = 100;
      else if(rand_val < 95)
         time_delta = 500;
      else
         time_delta = 1000;
      
      current_time += time_delta / 1000;  // конвертировать в секунды
      
      ticks[i].time = current_time;
      ticks[i].bid = NormalizeDouble(current_price, _Digits);
      ticks[i].ask = NormalizeDouble(current_price + spread, _Digits);
      ticks[i].last = 0;
      ticks[i].volume = 0;
      ticks[i].flags = 0;
   }
   
   return count;
}

//+------------------------------------------------------------------+
//| Проверить корректность декомпрессии                              |
//+------------------------------------------------------------------+
int CheckCorrectness(const MqlTick &original[], const MqlTick &decompressed[], int count)
{
   int errors = 0;
   
   for(int i = 0; i < count; i++)
   {
      // Проверить время
      if(original[i].time != decompressed[i].time)
      {
         if(errors < 5)  // показать первые 5 ошибок
            Print("Time error at #", i, ": ", original[i].time, " vs ", decompressed[i].time);
         errors++;
         continue;
      }
      
      // Проверить bid (с учетом точности)
      double bid_diff = MathAbs(original[i].bid - decompressed[i].bid);
      if(bid_diff > _Point * 0.1)  // допуск 0.1 пункта
      {
         if(errors < 5)
            Print("Bid error at #", i, ": ", original[i].bid, " vs ", decompressed[i].bid);
         errors++;
         continue;
      }
      
      // Проверить ask
      double ask_diff = MathAbs(original[i].ask - decompressed[i].ask);
      if(ask_diff > _Point * 0.1)
      {
         if(errors < 5)
            Print("Ask error at #", i, ": ", original[i].ask, " vs ", decompressed[i].ask);
         errors++;
         continue;
      }
   }
   
   return errors;
}
//+------------------------------------------------------------------+
