# Tick Compressor для MQL5 - Упрощенная версия

## 📋 Описание

Оптимизированный компрессор тиковых данных на основе:
- Результатов тестирования Python-прототипа
- Обсуждения на форуме MQL5
- Принципа: простота + эффективность

## ✨ Ключевые особенности

### ✅ Реализовано:
- **Fast Path (72% тиков)** - использует LUT, без умножений
- **Slow Path (27% тиков)** - переменная длина дельт (1-8 байт)
- **LUT оптимизация** - предвычисленные значения delta * _Point
- **Один ArrayResize** - вместо тысяч в цикле
- **Правильная работа с signed/unsigned** - для отрицательных дельт

### ❌ Намеренно НЕ реализовано:
- **RLE кодирование** - дает только 0.7%, сложность не оправдана
- **ZIP компрессия** - замедляет в 2 раза
- **Huffman кодирование** - сложность vs эффект

## 📊 Ожидаемая производительность

На основе Python-тестов + MQL5 компиляция:

| Метрика | Значение |
|---------|----------|
| Декомпрессия | **50-70 млн тиков/сек** |
| Сжатие | 5.7x от наивного хранения |
| Fast path | 72% тиков |
| Slow path | 27% тиков |
| Память на год EURUSD | ~550 MB |

## 🚀 Быстрый старт

### 1. Установка

Скопировать файлы в папку `MQL5/Include/`:
```
MQL5/Include/TickCompressor.mqh
MQL5/Scripts/TestCompressor.mq5
```

### 2. Базовое использование

```mql5
#include <TickCompressor.mqh>

void OnStart()
{
   // Загрузить тики
   MqlTick ticks[];
   int count = CopyTicks(_Symbol, ticks, COPY_TICKS_ALL, 0, 10000);
   
   // Создать компрессор
   CTickCompressor compressor;
   
   // Сжать
   uchar compressed[];
   int compressed_size = compressor.Compress(ticks, compressed, count);
   
   Print("Compressed ", count, " ticks to ", compressed_size, " bytes");
   
   // Распаковать
   MqlTick decompressed[];
   ArrayResize(decompressed, count);
   int restored = compressor.Decompress(compressed, decompressed, compressed_size);
   
   Print("Restored ", restored, " ticks");
   
   // Статистика
   compressor.PrintStats();
}
```

### 3. Сохранение в файл

```mql5
// Сжать и сохранить
uchar compressed[];
int size = compressor.Compress(ticks, compressed, count);

int handle = FileOpen("ticks_2024.dat", FILE_WRITE|FILE_BIN);
FileWriteArray(handle, compressed);
FileClose(handle);

// Загрузить и распаковать
handle = FileOpen("ticks_2024.dat", FILE_READ|FILE_BIN);
int file_size = (int)FileSize(handle);
ArrayResize(compressed, file_size);
FileReadArray(handle, compressed);
FileClose(handle);

MqlTick restored[];
ArrayResize(restored, 100000);  // примерный размер
int tick_count = compressor.Decompress(compressed, restored, file_size);
```

## 🔧 Архитектура

### Структура сжатых данных

```
[HEADER: 24 байта]
  - time (8 байт) - первый тик полностью
  - bid (8 байт)
  - ask (8 байт)

[TICK DATA] - для каждого следующего тика:

Fast Path (флаг 0x00):
  - flags (1 байт) = 0x00
  - bid_delta (1 байт signed)
  - ask_delta (1 байт signed)
  - time_delta (1 байт unsigned)
  Итого: 4 байта на тик

Slow Path (флаг 0x80):
  - flags (1 байт) = 0x8X (биты содержат размеры дельт)
  - bid_delta (1-8 байт)
  - ask_delta (1-8 байт)
  - time_delta (1-8 байт)
  Итого: 4-25 байт на тик
```

### Lookup Table (LUT)

```mql5
// Инициализация (один раз):
double g_delta_lut[512];
for(int i = 0; i < 512; i++) {
    int delta = i - 256;
    g_delta_lut[i] = delta * _Point;  // предвычислить!
}

// Использование в декомпрессии (быстро!):
// Вместо: new_bid = last_bid + delta * _Point;  // медленно
new_bid = last_bid + g_delta_lut[delta + 256];   // быстро!
```

## ⚡ Критические оптимизации

### 1. Предвыделение памяти

```mql5
// ПЛОХО:
for(int i = 0; i < count; i++) {
    ArrayResize(buffer, ArraySize(buffer) + 1);  // O(n²)
}

// ХОРОШО:
ArrayResize(buffer, max_size);  // один раз
int pos = 0;
for(int i = 0; i < count; i++) {
    buffer[pos++] = data[i];  // O(n)
}
ArrayResize(buffer, pos);  // уменьшить в конце
```

### 2. Fast Path первым

```mql5
// Проверять Fast Path ПЕРВЫМ (72% случаев!)
if((flags & 0x80) == 0) {
    // Fast path - выполняется в 72% случаев
    // LUT, никаких умножений
} else {
    // Slow path - только 27%
}
```

### 3. NormalizeDouble обязателен

```mql5
// ВСЕГДА нормализовать при декодировании
tick.bid = NormalizeDouble(last_bid + delta * _Point, _Digits);

// Без нормализации - микроошибки накапливаются!
```

## 📈 Сравнение с форумом

| Характеристика | Форум | Наша версия | Комментарий |
|---------------|-------|-------------|-------------|
| Декомпрессия | 55-59 млн/с | 50-70 млн/с (ожидаемо) | Сопоставимо |
| Сжатие BidAsk | 3.5x | 5.7x | ✓ Лучше |
| Сложность кода | Высокая | Низкая | ✓ Проще |
| RLE | Нет | Нет | Не нужен |
| LUT | Упомянут | Реализован | ✓ Ключ к скорости |
| Размер блока | 300K-600K | Гибко | Настраивается |

## 🎯 Интеграция в оптимизацию

### Предзагрузка данных

```mql5
// В OnTesterInit - один раз на всю оптимизацию
class CTickCache {
private:
    static MqlTick m_ticks[];
    static bool m_loaded;
    
public:
    static void LoadYear(int year) {
        if(m_loaded) return;
        
        // Загрузить сжатый файл
        uchar compressed[];
        LoadCompressedFile("ticks_" + year + ".dat", compressed);
        
        // Распаковать в память
        CTickCompressor compressor;
        ArrayResize(m_ticks, 10000000);  // 10M тиков
        int count = compressor.Decompress(compressed, m_ticks, ArraySize(compressed));
        ArrayResize(m_ticks, count);
        
        m_loaded = true;
        Print("Loaded ", count, " ticks for year ", year);
    }
    
    static void GetRange(datetime from, datetime to, MqlTick &result[]) {
        // Бинарный поиск + ArrayCopy
        // Все агенты читают из общего кэша - быстро!
    }
};

// Использование в эксперте
void OnTesterInit() {
    CTickCache::LoadYear(2024);  // один раз
}

void OnTester() {
    MqlTick ticks[];
    CTickCache::GetRange(start, end, ticks);  // быстро, без декомпрессии
    // тестировать стратегию...
}
```

## 🐛 Известные ограничения

1. **Только Bid/Ask** - Volume и Last не сжимаются (легко добавить)
2. **Отрицательные time_delta** - не поддерживаются (тики должны быть по порядку)
3. **Точность времени** - миллисекунды теряются (используется секунды)
4. **Максимальная дельта времени** - 255 сек для Fast Path

## 📝 Расширения (опционально)

### Добавить Volume

```mql5
// В Fast Path добавить:
compressed[pos++] = (uchar)volume;  // если volume < 256

// Или отдельно сжимать Volume Dictionary Encoding
```

### Адаптивный размер блока

```mql5
int GetOptimalBlockSize() {
    double volatility = CalculateVolatility();
    
    if(volatility > 0.001)  return 200000;  // высокая
    if(volatility > 0.0005) return 400000;  // средняя
    return 600000;  // низкая
}
```

### Добавить ZIP второго уровня

```mql5
// После сжатия дельт применить ZIP
// Замедлит в ~2 раза, но даст +30-50% сжатия
```

## 🧪 Тестирование

Запустить тестовый скрипт:
```
1. Открыть MetaEditor
2. Скомпилировать TestCompressor.mq5
3. Запустить в MetaTrader 5
4. Проверить лог:
   - Степень сжатия
   - Скорость декомпрессии
   - Корректность восстановления
```

Ожидаемый вывод:
```
=== Test Tick Compressor ===
Loaded 10000 real ticks
Compression ratio: 5.7x
Decompression speed: 65.4 million ticks/sec
Fast path: 72.1%
Slow path: 27.2%
SUCCESS: All ticks restored correctly!
```

## 💡 Советы по использованию

1. **Начни с малого** - протестируй на 10K тиков
2. **Измерь реальную производительность** - каждый символ уникален
3. **Сравни с текущим решением** - возможно, оно уже оптимально
4. **Профилируй** - возможно, узкое место не в декомпрессии
5. **Кэшируй** - распаковывай один раз, используй многократно

## 🔗 Ссылки

- [Обсуждение на форуме MQL5](https://www.mql5.com/ru/forum/499639)
- [Python прототип](tick_compressor.py)
- [Результаты тестирования](SUMMARY.md)

## 📞 Вопросы?

Если что-то непонятно или нужна помощь с интеграцией - пиши!

---

**Версия:** 1.0  
**Дата:** 2024-11-18  
**Лицензия:** Используй как хочешь, главное чтобы работало 😊
