#!/usr/bin/env python3
"""
Balanced Tick Compressor - Python прототип для MQL5
Использует NumPy для максимального приближения к MQL5 массивам
"""

import numpy as np
import time
from typing import Tuple
import struct

# Определяем структуру тика (аналог MqlTick в MQL5)
tick_dtype = np.dtype([
    ('time', np.int64),      # datetime в MQL5 = long
    ('bid', np.float64),     # double
    ('ask', np.float64),     # double
    ('last', np.float64),    # double (не используем пока)
    ('volume', np.uint64),   # ulong (не используем пока)
    ('flags', np.uint32)     # uint (не используем пока)
])


class BalancedTickCompressor:
    """
    Балансированный компрессор тиков с оптимизациями:
    1. LUT (Lookup Tables) для малых дельт
    2. Fast path для типичных тиков
    3. RLE для нулевых дельт
    4. Битовые флаги для метаданных
    """
    
    # Константы
    LUT_SIZE = 512
    LUT_HALF = 256
    FAST_PATH_THRESHOLD = 8  # дельты от -8 до +8
    
    # Флаги в заголовке
    FLAG_FAST_PATH = 0x00
    FLAG_RLE = 0x20
    FLAG_SPREAD_ZERO = 0x10
    
    def __init__(self, point: float = 0.00001):
        """
        Инициализация компрессора
        
        Args:
            point: Размер пункта (для EURUSD обычно 0.00001)
        """
        self.point = point
        self.digits = self._calculate_digits(point)
        
        # LUT для bid и ask (аналог статических массивов в MQL5)
        self.bid_lut = np.zeros(self.LUT_SIZE, dtype=np.float64)
        self.ask_lut = np.zeros(self.LUT_SIZE, dtype=np.float64)
        
        # Статистика
        self.stats = {
            'fast_path_count': 0,
            'slow_path_count': 0,
            'rle_count': 0,
            'total_ticks': 0
        }
    
    def _calculate_digits(self, point: float) -> int:
        """Вычислить количество знаков после запятой"""
        if point >= 0.01:
            return 2
        elif point >= 0.001:
            return 3
        elif point >= 0.0001:
            return 4
        else:
            return 5
    
    def _update_lut(self, base_bid: float, base_ask: float):
        """
        Обновить Lookup Tables для дельт
        LUT хранит предвычисленные значения delta * point
        """
        indices = np.arange(-self.LUT_HALF, self.LUT_HALF, dtype=np.int32)
        # Сохраняем дельты, а не абсолютные значения
        self.bid_lut = indices * self.point
        self.ask_lut = indices * self.point
    
    def _normalize_double(self, value: float) -> float:
        """
        Аналог NormalizeDouble в MQL5
        """
        return round(value, self.digits)
    
    def compress(self, ticks: np.ndarray) -> bytearray:
        """
        Сжать массив тиков
        
        Args:
            ticks: NumPy массив с dtype=tick_dtype
            
        Returns:
            bytearray со сжатыми данными
        """
        if len(ticks) == 0:
            return bytearray()
        
        compressed = bytearray()
        
        # Записываем заголовок: первый тик полностью
        first_tick = ticks[0]
        compressed.extend(struct.pack('<q', first_tick['time']))  # long
        compressed.extend(struct.pack('<d', first_tick['bid']))   # double
        compressed.extend(struct.pack('<d', first_tick['ask']))   # double
        
        # Обновляем LUT на основе первого тика
        self._update_lut(first_tick['bid'], first_tick['ask'])
        
        # Сжимаем остальные тики
        last_bid = first_tick['bid']
        last_ask = first_tick['ask']
        last_time = first_tick['time']
        
        i = 1
        while i < len(ticks):
            current = ticks[i]
            
            # Вычисляем дельты
            bid_delta = int(round((current['bid'] - last_bid) / self.point))
            ask_delta = int(round((current['ask'] - last_ask) / self.point))
            time_delta = int(current['time'] - last_time)
            
            # Проверка на RLE (серия одинаковых тиков)
            if bid_delta == 0 and ask_delta == 0 and i + 1 < len(ticks):
                rle_count = 1
                first_time_delta = time_delta  # Сохраняем первую дельту
                j = i + 1
                while j < len(ticks) and rle_count < 65535:
                    next_tick = ticks[j]
                    next_bid_delta = int(round((next_tick['bid'] - last_bid) / self.point))
                    next_ask_delta = int(round((next_tick['ask'] - last_ask) / self.point))
                    next_time_delta = int(next_tick['time'] - ticks[j-1]['time'])
                    
                    # Проверяем, что цена не изменилась И время дельта такая же
                    if next_bid_delta == 0 and next_ask_delta == 0 and next_time_delta == first_time_delta:
                        rle_count += 1
                        j += 1
                    else:
                        break
                
                if rle_count >= 3:  # RLE эффективен только для 3+ повторов
                    # Записываем RLE блок
                    compressed.append(self.FLAG_RLE)
                    compressed.extend(struct.pack('<H', rle_count))  # количество
                    compressed.extend(struct.pack('<H', first_time_delta))  # дельта времени
                    
                    self.stats['rle_count'] += rle_count
                    last_time = ticks[i + rle_count - 1]['time']
                    i += rle_count
                    continue
            
            # Fast path: малые дельты, которые попадают в LUT
            if (abs(bid_delta) <= self.FAST_PATH_THRESHOLD and 
                abs(ask_delta) <= self.FAST_PATH_THRESHOLD and
                time_delta <= 255):
                
                compressed.append(self.FLAG_FAST_PATH)
                compressed.extend(struct.pack('b', bid_delta))  # signed char
                compressed.extend(struct.pack('b', ask_delta))  # signed char
                compressed.append(np.uint8(time_delta))  # uchar
                
                self.stats['fast_path_count'] += 1
            
            # Slow path: большие дельты
            else:
                # Флаги для размера дельт
                flags = 0x80  # старший бит = slow path
                
                # Определяем размер для каждой дельты
                bid_bytes = self._get_delta_size(bid_delta)
                ask_bytes = self._get_delta_size(ask_delta)
                time_bytes = self._get_delta_size(time_delta)
                
                flags |= (bid_bytes << 4)  # биты 4-5
                flags |= (ask_bytes << 2)  # биты 2-3
                flags |= time_bytes        # биты 0-1
                
                compressed.append(flags)
                
                # Записываем дельты переменной длины
                self._write_delta(compressed, bid_delta, bid_bytes)
                self._write_delta(compressed, ask_delta, ask_bytes)
                self._write_delta(compressed, time_delta, time_bytes)
                
                self.stats['slow_path_count'] += 1
            
            # Обновляем последние значения
            last_bid = current['bid']
            last_ask = current['ask']
            last_time = current['time']
            i += 1
        
        self.stats['total_ticks'] = len(ticks)
        return compressed
    
    def _get_delta_size(self, delta: int) -> int:
        """
        Определить размер дельты в байтах
        00 = 1 байт (-128..127)
        01 = 2 байта (-32768..32767)
        10 = 4 байта
        11 = 8 байтов
        """
        if -128 <= delta <= 127:
            return 0
        elif -32768 <= delta <= 32767:
            return 1
        elif -2147483648 <= delta <= 2147483647:
            return 2
        else:
            return 3
    
    def _write_delta(self, buffer: bytearray, delta: int, size_code: int):
        """Записать дельту переменной длины"""
        if size_code == 0:
            buffer.extend(struct.pack('b', delta))  # signed char
        elif size_code == 1:
            buffer.extend(struct.pack('<h', np.int16(delta)))
        elif size_code == 2:
            buffer.extend(struct.pack('<i', np.int32(delta)))
        else:
            buffer.extend(struct.pack('<q', np.int64(delta)))
    
    def decompress(self, compressed: bytearray, max_ticks: int = -1) -> np.ndarray:
        """
        Распаковать сжатые данные
        
        Args:
            compressed: Сжатые данные
            max_ticks: Максимальное количество тиков (-1 = все)
            
        Returns:
            NumPy массив тиков
        """
        if len(compressed) < 24:  # минимальный размер заголовка
            return np.array([], dtype=tick_dtype)
        
        # Создаем массив для результата (аналог MqlTick ticks[] в MQL5)
        result = []
        
        # Читаем заголовок
        pos = 0
        first_time = struct.unpack('<q', compressed[pos:pos+8])[0]
        pos += 8
        first_bid = struct.unpack('<d', compressed[pos:pos+8])[0]
        pos += 8
        first_ask = struct.unpack('<d', compressed[pos:pos+8])[0]
        pos += 8
        
        # Первый тик
        tick = np.zeros(1, dtype=tick_dtype)[0]
        tick['time'] = first_time
        tick['bid'] = first_bid
        tick['ask'] = first_ask
        result.append(tick)
        
        # Обновляем LUT
        self._update_lut(first_bid, first_ask)
        
        last_bid = first_bid
        last_ask = first_ask
        last_time = first_time
        
        tick_count = 1
        
        # Распаковываем остальные тики
        while pos < len(compressed) and (max_ticks == -1 or tick_count < max_ticks):
            flags = compressed[pos]
            pos += 1
            
            # RLE блок
            if flags == self.FLAG_RLE:
                if pos + 4 > len(compressed):
                    break
                rle_count = struct.unpack('<H', compressed[pos:pos+2])[0]
                pos += 2
                time_delta = struct.unpack('<H', compressed[pos:pos+2])[0]
                pos += 2
                
                # Создаем rle_count одинаковых тиков с одинаковой временной дельтой
                for j in range(rle_count):
                    tick = np.zeros(1, dtype=tick_dtype)[0]
                    tick['time'] = last_time + time_delta * (j + 1)
                    tick['bid'] = last_bid
                    tick['ask'] = last_ask
                    result.append(tick)
                    tick_count += 1
                    
                    if max_ticks != -1 and tick_count >= max_ticks:
                        break
                
                # Обновляем last_time на конец RLE блока
                last_time = tick['time']
                continue
            
            # Fast path
            if flags == self.FLAG_FAST_PATH:
                if pos + 3 > len(compressed):
                    break
                    
                bid_delta = struct.unpack('b', compressed[pos:pos+1])[0]  # signed char
                pos += 1
                ask_delta = struct.unpack('b', compressed[pos:pos+1])[0]  # signed char
                pos += 1
                time_delta = compressed[pos]
                pos += 1
                
                # Используем LUT для быстрого декодирования дельт
                tick = np.zeros(1, dtype=tick_dtype)[0]
                tick['bid'] = self._normalize_double(last_bid + self.bid_lut[bid_delta + self.LUT_HALF])
                tick['ask'] = self._normalize_double(last_ask + self.ask_lut[ask_delta + self.LUT_HALF])
                tick['time'] = last_time + time_delta
                
                result.append(tick)
                
                last_bid = tick['bid']
                last_ask = tick['ask']
                last_time = tick['time']
                tick_count += 1
                continue
            
            # Slow path
            if flags & 0x80:
                bid_size = (flags >> 4) & 0x03
                ask_size = (flags >> 2) & 0x03
                time_size = flags & 0x03
                
                # Читаем дельты
                bid_delta = self._read_delta(compressed, pos, bid_size)
                pos += (1 << bid_size)
                
                ask_delta = self._read_delta(compressed, pos, ask_size)
                pos += (1 << ask_size)
                
                time_delta = self._read_delta(compressed, pos, time_size)
                pos += (1 << time_size)
                
                # Декодируем цены
                tick = np.zeros(1, dtype=tick_dtype)[0]
                tick['bid'] = self._normalize_double(last_bid + bid_delta * self.point)
                tick['ask'] = self._normalize_double(last_ask + ask_delta * self.point)
                tick['time'] = last_time + time_delta
                
                result.append(tick)
                
                last_bid = tick['bid']
                last_ask = tick['ask']
                last_time = tick['time']
                tick_count += 1
        
        # Конвертируем список в NumPy массив
        if len(result) == 0:
            return np.array([], dtype=tick_dtype)
        
        # Создаем массив и копируем данные (аналог ArrayCopy в MQL5)
        ticks_array = np.zeros(len(result), dtype=tick_dtype)
        for i, t in enumerate(result):
            ticks_array[i] = t
        
        return ticks_array
    
    def _read_delta(self, buffer: bytearray, pos: int, size_code: int) -> int:
        """Прочитать дельту переменной длины"""
        if size_code == 0:
            return struct.unpack('b', buffer[pos:pos+1])[0]  # signed char
        elif size_code == 1:
            return struct.unpack('<h', buffer[pos:pos+2])[0]
        elif size_code == 2:
            return struct.unpack('<i', buffer[pos:pos+4])[0]
        else:
            return struct.unpack('<q', buffer[pos:pos+8])[0]
    
    def get_compression_ratio(self, original_size: int, compressed_size: int) -> float:
        """Вычислить степень сжатия"""
        if compressed_size == 0:
            return 0.0
        return original_size / compressed_size
    
    def print_stats(self):
        """Вывести статистику"""
        total = self.stats['total_ticks']
        if total == 0:
            return
        
        print("\n=== Статистика сжатия ===")
        print(f"Всего тиков: {total}")
        print(f"Fast path: {self.stats['fast_path_count']} ({100*self.stats['fast_path_count']/total:.1f}%)")
        print(f"Slow path: {self.stats['slow_path_count']} ({100*self.stats['slow_path_count']/total:.1f}%)")
        print(f"RLE блоков: {self.stats['rle_count']} ({100*self.stats['rle_count']/total:.1f}%)")


def generate_synthetic_ticks(count: int, volatility: float = 0.0001) -> np.ndarray:
    """
    Генерировать синтетические тики для тестирования
    
    Args:
        count: Количество тиков
        volatility: Волатильность (стандартное отклонение изменений цены)
        
    Returns:
        NumPy массив тиков
    """
    ticks = np.zeros(count, dtype=tick_dtype)
    
    # Начальные значения
    base_price = 1.08500
    spread = 0.00010  # 1 пункт
    
    ticks[0]['time'] = 1700000000000  # начальное время в миллисекундах
    ticks[0]['bid'] = base_price
    ticks[0]['ask'] = base_price + spread
    
    # Генерируем изменения цены
    np.random.seed(42)
    price_changes = np.random.normal(0, volatility, count-1)
    
    # Время между тиками: обычно 10-1000 мс, иногда больше
    time_deltas = np.random.choice(
        [10, 50, 100, 500, 1000, 5000],
        size=count-1,
        p=[0.3, 0.3, 0.2, 0.1, 0.08, 0.02]
    )
    
    # Заполняем массив
    current_price = base_price
    current_time = ticks[0]['time']
    
    i = 1
    while i < count:
        # Иногда создаем RLE блоки (одинаковая цена и равномерное время)
        if i % 1000 == 0 and i + 10 < count:
            repeat_count = min(np.random.randint(5, 11), count - i)
            time_step = 100  # постоянная дельта времени
            
            for j in range(repeat_count):
                ticks[i + j]['time'] = current_time + j * time_step
                ticks[i + j]['bid'] = round(current_price, 5)
                ticks[i + j]['ask'] = round(current_price + spread, 5)
            
            i += repeat_count
            current_time = ticks[i - 1]['time']
        else:
            # Обычный тик
            current_price += price_changes[i-1]
            current_time += time_deltas[i-1]
            
            ticks[i]['time'] = current_time
            ticks[i]['bid'] = round(current_price, 5)
            ticks[i]['ask'] = round(current_price + spread, 5)
            i += 1
    
    return ticks


def test_compressor():
    """Тестирование компрессора"""
    print("=" * 60)
    print("ТЕСТ БАЛАНСИРОВАННОГО КОМПРЕССОРА ТИКОВ")
    print("=" * 60)
    
    # Генерируем тестовые данные
    print("\n1. Генерация синтетических данных...")
    tick_count = 100000  # 100K тиков (примерно 1 день для EURUSD)
    ticks = generate_synthetic_ticks(tick_count, volatility=0.00005)
    
    original_size = ticks.nbytes
    print(f"   Сгенерировано тиков: {tick_count}")
    print(f"   Размер в памяти: {original_size:,} байт ({original_size/1024/1024:.2f} MB)")
    
    # Инициализируем компрессор
    compressor = BalancedTickCompressor(point=0.00001)
    
    # Тест сжатия
    print("\n2. Сжатие данных...")
    start_time = time.time()
    compressed = compressor.compress(ticks)
    compress_time = time.time() - start_time
    
    compressed_size = len(compressed)
    compression_ratio = compressor.get_compression_ratio(original_size, compressed_size)
    
    print(f"   Сжатый размер: {compressed_size:,} байт ({compressed_size/1024/1024:.2f} MB)")
    print(f"   Степень сжатия: {compression_ratio:.2f}x")
    print(f"   Время сжатия: {compress_time:.3f} сек")
    print(f"   Скорость: {tick_count/compress_time/1_000_000:.2f} млн тиков/сек")
    
    # Тест распаковки
    print("\n3. Распаковка данных...")
    start_time = time.time()
    decompressed = compressor.decompress(compressed)
    decompress_time = time.time() - start_time
    
    print(f"   Распаковано тиков: {len(decompressed)}")
    print(f"   Время распаковки: {decompress_time:.3f} сек")
    print(f"   Скорость: {len(decompressed)/decompress_time/1_000_000:.2f} млн тиков/сек")
    
    # Проверка корректности
    print("\n4. Проверка корректности...")
    errors = 0
    for i in range(min(len(ticks), len(decompressed))):
        if (ticks[i]['time'] != decompressed[i]['time'] or
            abs(ticks[i]['bid'] - decompressed[i]['bid']) > 1e-5 or
            abs(ticks[i]['ask'] - decompressed[i]['ask']) > 1e-5):
            errors += 1
            if errors <= 5:  # показываем первые 5 ошибок
                print(f"   Ошибка в тике #{i}:")
                print(f"      Оригинал: time={ticks[i]['time']}, bid={ticks[i]['bid']}, ask={ticks[i]['ask']}")
                print(f"      Распаковка: time={decompressed[i]['time']}, bid={decompressed[i]['bid']}, ask={decompressed[i]['ask']}")
    
    if errors == 0:
        print("   ✓ Все тики восстановлены корректно!")
    else:
        print(f"   ✗ Найдено ошибок: {errors} из {len(ticks)}")
    
    # Статистика
    compressor.print_stats()
    
    # Сравнение с "наивным" подходом
    print("\n=== Сравнение с наивным хранением ===")
    naive_size = tick_count * (8 + 8 + 8)  # time + bid + ask
    print(f"Наивное хранение: {naive_size:,} байт ({naive_size/1024/1024:.2f} MB)")
    print(f"Наше сжатие: {compressed_size:,} байт ({compressed_size/1024/1024:.2f} MB)")
    print(f"Выигрыш: {naive_size/compressed_size:.2f}x")
    
    print("\n" + "=" * 60)
    print("ТЕСТ ЗАВЕРШЕН")
    print("=" * 60)


if __name__ == "__main__":
    test_compressor()
