//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
#property copyright   "Portfolio Manager - by transcendreamer"
#property description "Expert for Portfolio Modeller indicator"

#include <MT4Bridge/MT4Orders.mqh>
#include <MT4Bridge/ind4to5.mqh>
#include <MT4Bridge/MT4Account.mqh>
#include <MT4Bridge/MT4Time.mqh>
#include <MT4Bridge/MT4MarketInfo.mqh>
#include <MT4Bridge/MT4State.mqh>
#include <ErrorDescription.mqh>


//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
#define MAX_SYMBOLS  32
#define FONT_TEXT    "Verdana"
#define FONT_BUTTON  "Wingdings"
#define SIZE_TEXT    9
#define SIZE_BUTTON  14
#define SPACING      14
#define XMARGIN      5
#define YMARGIN      25
#define SPECIAL      '='
#define CHAR_BUY     221
#define CHAR_SELL    222
#define CHAR_CLOSE   253
#define CHAR_TRANS   91
#define CHAR_SAVE    61
#define XBUTTON1     5
#define XBUTTON2     25
#define XBUTTON3     45
#define XBUTTON4     65
#define XBUTTON5     85
#define YBUTTON      3
#define KEY_SELL     219
#define KEY_BUY      221
#define KEY_CLOSE    220
#define KEY_TRANS    191
#define KEY_SAVE     192
#define LINES_COLOR  clrGreen
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
extern string           ______GENERAL_______="______GENERAL_______";
extern string           Portfolio_Name="";
extern int              Magic_Number=0;
extern bool             Terminal_Mode=false;
extern string           Comment_Filter="";

extern string           ______TRADING______="______TRADING______";
extern double           Take_Profit=0;
extern double           Stop_Loss=0;
extern double           Breakeven_Level=0;
enum   ENUM_SCALING     {no_scaling,last_volume,full_volume,next_volume,extra_volume};
extern ENUM_SCALING     Scaling_Profit=no_scaling;
extern ENUM_SCALING     Scaling_Loss=no_scaling;
extern ENUM_SCALING     Scaling_Breakeven=no_scaling;
extern int              Lots_Digits=2;
extern double           Max_Volume=0;

extern string           ______AUTOMAT______="______AUTOMAT______";
extern datetime         Starting_From=D'2012.01.01 00:00';
extern datetime         Finishing_From=D'2035.01.01 00:00';
extern bool             Use_Lines=false;
extern bool             Use_Adding=false;
extern bool             Use_Correction=false;
extern int              Iteration_Minutes=60;
extern double           Adding_Threshold=0;
extern double           Correction_Profit=0;
extern double           Correction_Loss=0;
extern double           Gap_Protect=100;

extern string           ______ROLLOVER______="______ROLLOVER______";
extern int              Stop_Hour=23;
extern int              Stop_Minute=45;
extern int              Resume_Hour=00;
extern int              Resume_Minute=25;

extern string           ______FILTERS______="______FILTERS______";
enum   ENUM_FILTER      {zero,model,main,fast,slow,upper,lower};
extern bool             Filter_Quality=0;
extern ENUM_FILTER      Filter_Line_Buy=zero;
extern ENUM_FILTER      Filter_Line_Sell=zero;
enum   ENUM_OPERATOR    {never,above,below,always};
extern ENUM_OPERATOR    Filter_Op_Buy=never;
extern ENUM_OPERATOR    Filter_Op_Sell=never;
enum   ENUM_REVERSAL    {reverse,close,ignore};
extern ENUM_REVERSAL    Reversal_Signal=ignore;
enum   ENUM_PROGRESS    {equal,fibo,martin,linear};
extern ENUM_PROGRESS    Volume_Progression=equal;

extern string           ______TRAILING______="______TRAILING______";
extern bool             Use_Trailing=false;
extern ENUM_FILTER      Trailing_Long=zero;
extern ENUM_FILTER      Trailing_Short=zero;
extern ENUM_FILTER      Condition_Long=zero;
extern ENUM_FILTER      Condition_Short=zero;

extern string           ______OTHER______="______OTHER______";
extern color            Text_Color=clrMagenta;
extern ENUM_BASE_CORNER Text_Corner=CORNER_LEFT_LOWER;
extern bool             Show_Levels=true;
extern bool             Show_Trades=false;
extern bool             Auto_Alerts=false;
extern bool             Push_Notifications=false;
extern bool             Manual_Confirm=true;
extern int              Retry_Delay=500;
extern int              Retry_Times=50;
extern bool             Template_Autosave=true;
extern int              Screenshot_Seconds=0;
extern int              Screenshot_Width=640;
extern int              Screenshot_Height=480;
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool     error;
int      window;
long     chart;
string   currency;
int      total,increment;
bool     breakeven;
string   SYMBOLS[MAX_SYMBOLS];
double   LOTS[MAX_SYMBOLS];
double   profit,volume;
double   limit_profit,limit_loss,limit_breakeven;
double   previous=EMPTY_VALUE,current=EMPTY_VALUE;
double   threshold,level_buy,level_sell,quality;
datetime iteration=0,screen_timer=0;
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void OnInit()
  {
   SetupInitial();
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
   CleanAll();
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void SetupInitial()
  {
   CleanAll();

   error=false;
   chart=ChartID();
   window=ObjectFind("Data-label-"+Portfolio_Name+"-1");
   currency=AccountCurrency();

   if(Portfolio_Name=="")
     {
      error=true;
      MessageBox("Empty portfolio name!","",MB_ICONERROR);
      return;
     }
   if(window<0)
     {
      error=true;
      MessageBox("Portfolio "+Portfolio_Name+" not found!","",MB_ICONERROR);
      return;
     }

   PlaceLabel("Button_Close",XBUTTON1,YBUTTON,Text_Corner,CharToString(CHAR_CLOSE),Text_Color,FONT_BUTTON,SIZE_BUTTON);
   PlaceLabel("Button_Buy",XBUTTON2,YBUTTON,Text_Corner,CharToString(CHAR_BUY),Text_Color,FONT_BUTTON,SIZE_BUTTON);
   PlaceLabel("Button_Sell",XBUTTON3,YBUTTON,Text_Corner,CharToString(CHAR_SELL),Text_Color,FONT_BUTTON,SIZE_BUTTON);
   PlaceLabel("Button_Trans",XBUTTON4,YBUTTON,Text_Corner,CharToString(CHAR_TRANS),Text_Color,FONT_BUTTON,SIZE_BUTTON);
   PlaceLabel("Button_Save",XBUTTON5,YBUTTON,Text_Corner,CharToString(CHAR_SAVE),Text_Color,FONT_BUTTON,SIZE_BUTTON);

   RestoreGlobals();
   ReadScreenFormula();
   OnTick();
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CleanAll()
  {
   for(int n=ObjectsTotal(); n>=1; n--)
     {
      string name=ObjectName(n-1);
      if(StringFind(name,"Status-line-"+Portfolio_Name)!=-1)
         ObjectDelete(name);
      if(StringFind(name,"Stop-"+Portfolio_Name)!=-1)
         ObjectDelete(name);
      if(StringFind(name,"Take-"+Portfolio_Name)!=-1)
         ObjectDelete(name);
      if(StringFind(name,"Zero-"+Portfolio_Name)!=-1)
         ObjectDelete(name);
      if(StringFind(name,"Safe-"+Portfolio_Name)!=-1)
         ObjectDelete(name);
     }

   ObjectDelete("Button_Sell");
   ObjectDelete("Button_Buy");
   ObjectDelete("Button_Close");
   ObjectDelete("Button_Trans");
   ObjectDelete("Button_Save");

   GlobalVariableDel("Volume-"+Portfolio_Name);
   GlobalVariableDel("Iteration-"+Portfolio_Name);
   GlobalVariableDel("Threshold-"+Portfolio_Name);
   GlobalVariableDel("Increment-"+Portfolio_Name);
   GlobalVariableDel("Breakeven-"+Portfolio_Name);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void OnChartEvent(const int id,const long &lparam,const double &dparam,const string &sparam)
  {
   if(error)
      return;

   bool buy=false,sell=false,close=false,trans=false,save=false;

   if(id==CHARTEVENT_OBJECT_CLICK)
     {
      if(sparam=="Button_Buy")
         buy=true;
      if(sparam=="Button_Sell")
         sell=true;
      if(sparam=="Button_Close")
         close=true;
      if(sparam=="Button_Trans")
         trans=true;
      if(sparam=="Button_Save")
         save=true;
     }

   if(id==CHARTEVENT_KEYDOWN)
     {
      if(int(lparam)==KEY_BUY)
         buy=true;
      if(int(lparam)==KEY_SELL)
         sell=true;
      if(int(lparam)==KEY_CLOSE)
         close=true;
      if(int(lparam)==KEY_TRANS)
         trans=true;
      if(int(lparam)==KEY_SAVE)
         save=true;
     }

   if(buy)
     {
      Print(Portfolio_Name+": Manual BUY");
      if(DoOpen(GetVolume(increment+1),false,increment+1))
        {
         UpdateStatus();
         DoSave(false);
        }
     }

   if(sell)
     {
      Print(Portfolio_Name+": Manual SELL");
      if(DoOpen(-GetVolume(increment+1),false,increment+1))
        {
         UpdateStatus();
         DoSave(false);
        }
     }

   if(close)
     {
      Print(Portfolio_Name+": Manual CLOSE");
      if(DoClose(false,0))
        {
         UpdateStatus();
         DoSave(false);
        }
     }

   if(trans)
     {
      Print(Portfolio_Name+": Manual TRANS");
      if(DoTrans(volume,false,0))
        {
         UpdateStatus();
         DoSave(false);
        }
     }

   if(save)
     {
      DoSave(false);
     }

  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void OnTick()
  {
   if(error)
      return;
   UpdateStatus();
   CheckLimits();
   CheckIterations();
   CheckLines();
   MakeScreenshot(0);
   previous=current;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void UpdateStatus()
  {

   current=GlobalVariableGet("Portfolio-"+Portfolio_Name);
   quality=GlobalVariableGet("Quality-"+Portfolio_Name);
   threshold=GlobalVariableGet("Threshold-"+Portfolio_Name);
   increment=(int)GlobalVariableGet("Increment-"+Portfolio_Name);
   iteration=(int)GlobalVariableGet("Iteration-"+Portfolio_Name);
   breakeven=(bool)GlobalVariableGet("Breakeven-"+Portfolio_Name);
   volume=GlobalVariableGet("Volume-"+Portfolio_Name);
   profit=GetProfit(0);

   if(Terminal_Mode)
      if(profit!=0 && volume==0)
         volume=1;


   switch(Filter_Line_Buy)
     {
      case main:
         level_buy=GlobalVariableGet("Main-"+Portfolio_Name);
         break;
      case fast:
         level_buy=GlobalVariableGet("Fast-"+Portfolio_Name);
         break;
      case slow:
         level_buy=GlobalVariableGet("Slow-"+Portfolio_Name);
         break;
      case upper:
         level_buy=GlobalVariableGet("Upper-"+Portfolio_Name);
         break;
      case lower:
         level_buy=GlobalVariableGet("Lower-"+Portfolio_Name);
         break;
      case model:
         level_buy=GlobalVariableGet("Model-"+Portfolio_Name);
         break;
      default:
         level_buy=0;
         break;
     }


   switch(Filter_Line_Sell)
     {
      case main:
         level_sell=GlobalVariableGet("Main-"+Portfolio_Name);
         break;
      case fast:
         level_sell=GlobalVariableGet("Fast-"+Portfolio_Name);
         break;
      case slow:
         level_sell=GlobalVariableGet("Slow-"+Portfolio_Name);
         break;
      case upper:
         level_sell=GlobalVariableGet("Upper-"+Portfolio_Name);
         break;
      case lower:
         level_sell=GlobalVariableGet("Lower-"+Portfolio_Name);
         break;
      case model:
         level_sell=GlobalVariableGet("Model-"+Portfolio_Name);
         break;
      default:
         level_sell=0;
         break;
     }


   if(breakeven)
      limit_loss=0;
   else
      switch(Scaling_Loss)
        {
         case no_scaling:
            limit_loss=Stop_Loss;
            break;
         case last_volume:
            limit_loss=Stop_Loss*GetVolume(increment);
            break;
         case next_volume:
            limit_loss=Stop_Loss*GetVolume(increment+1);
            break;
         case full_volume:
            limit_loss=Stop_Loss*MathAbs(volume);
            break;
         case extra_volume:
            limit_loss=Stop_Loss*(MathAbs(volume)+GetVolume(increment+1));
            break;
         default:
            limit_loss=Stop_Loss;
            break;
        }


   switch(Scaling_Profit)
     {
      case no_scaling:
         limit_profit=Take_Profit;
         break;
      case last_volume:
         limit_profit=Take_Profit*GetVolume(increment);
         break;
      case next_volume:
         limit_profit=Take_Profit*GetVolume(increment+1);
         break;
      case full_volume:
         limit_profit=Take_Profit*MathAbs(volume);
         break;
      case extra_volume:
         limit_profit=Take_Profit*(MathAbs(volume)+GetVolume(increment+1));
         break;
      default:
         limit_profit=Take_Profit;
         break;
     }


   switch(Scaling_Breakeven)
     {
      case no_scaling:
         limit_breakeven=Breakeven_Level;
         break;
      case last_volume:
         limit_breakeven=Breakeven_Level*GetVolume(increment);
         break;
      case next_volume:
         limit_breakeven=Breakeven_Level*GetVolume(increment+1);
         break;
      case full_volume:
         limit_breakeven=Breakeven_Level*MathAbs(volume);
         break;
      case extra_volume:
         limit_breakeven=Breakeven_Level*(MathAbs(volume)+GetVolume(increment+1));
         break;
      default:
         limit_breakeven=Breakeven_Level;
         break;
     }


   int offset=YMARGIN;
   string text=("Name: "+Portfolio_Name);
   string name=("Status-line-"+Portfolio_Name+"-A");
   PlaceLabel(name,XMARGIN,offset,Text_Corner,text,Text_Color,FONT_TEXT,SIZE_TEXT);

   offset+=SPACING;
   text=("Profit: "+DoubleToStr(profit,2)+" "+currency);
   name=("Status-line-"+Portfolio_Name+"-B");
   PlaceLabel(name,XMARGIN,offset,Text_Corner,text,Text_Color,FONT_TEXT,SIZE_TEXT);

   offset+=SPACING;
   text=("Volume: "+DoubleToStr(volume,2));
   name=("Status-line-"+Portfolio_Name+"-C");
   PlaceLabel(name,XMARGIN,offset,Text_Corner,text,Text_Color,FONT_TEXT,SIZE_TEXT);


   if(Use_Adding || Use_Correction)
     {
      offset+=SPACING;
      if(iteration==0)
         text=("Iteration: N/A");
      else
         text=("Iteration: "+TimeToString(iteration,TIME_MINUTES|TIME_SECONDS));
      name=("Status-line-"+Portfolio_Name+"-D");
      PlaceLabel(name,XMARGIN,offset,Text_Corner,text,Text_Color,FONT_TEXT,SIZE_TEXT);

      offset+=SPACING;
      text=("Increment: "+IntegerToString(increment));
      name=("Status-line-"+Portfolio_Name+"-E");
      PlaceLabel(name,XMARGIN,offset,Text_Corner,text,Text_Color,FONT_TEXT,SIZE_TEXT);

      offset+=SPACING;
      text=("Trigger: "+DoubleToStr(threshold,2)+" "+currency);
      name=("Status-line-"+Portfolio_Name+"-F");
      PlaceLabel(name,XMARGIN,offset,Text_Corner,text,Text_Color,FONT_TEXT,SIZE_TEXT);
     }


   if(Show_Levels)
      if(volume!=0 && profit!=0)
        {
         if(true)
           {
            double zero=NormalizeDouble((current-profit/volume),2);
            PlaceHorizontal("Zero-"+Portfolio_Name,zero,LINES_COLOR,STYLE_DASHDOTDOT);
           }
         if(Stop_Loss!=0)
           {
            double stop=NormalizeDouble(current-(limit_loss+profit)/volume,2);
            PlaceHorizontal("Stop-"+Portfolio_Name,stop,LINES_COLOR,STYLE_DASHDOTDOT);
           }
         if(Take_Profit!=0)
           {
            double take=NormalizeDouble(current+(limit_profit-profit)/volume,2);
            PlaceHorizontal("Take-"+Portfolio_Name,take,LINES_COLOR,STYLE_DASHDOTDOT);
           }
         if(Breakeven_Level!=0)
           {
            double be=NormalizeDouble(current+(limit_breakeven-profit)/volume,2);
            PlaceHorizontal("Safe-"+Portfolio_Name,be,LINES_COLOR,STYLE_DASHDOTDOT);
           }
        }

  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void RestoreGlobals()
  {
   double v=StringToDouble(ObjectGetString(chart,"Volume-"+Portfolio_Name,OBJPROP_TEXT));
   double i=StringToDouble(ObjectGetString(chart,"Iteration-"+Portfolio_Name,OBJPROP_TEXT));
   double n=StringToDouble(ObjectGetString(chart,"Increment-"+Portfolio_Name,OBJPROP_TEXT));
   double t=StringToDouble(ObjectGetString(chart,"Threshold-"+Portfolio_Name,OBJPROP_TEXT));
   double b=StringToDouble(ObjectGetString(chart,"Breakeven-"+Portfolio_Name,OBJPROP_TEXT));
   GlobalVariableSet("Volume-"+Portfolio_Name,v);
   GlobalVariableSet("Iteration-"+Portfolio_Name,i);
   GlobalVariableSet("Increment-"+Portfolio_Name,v);
   GlobalVariableSet("Threshold-"+Portfolio_Name,i);
   GlobalVariableSet("Breakeven-"+Portfolio_Name,i);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void BackupGlobals()
  {
   string i=DoubleToString(GlobalVariableGet("Iteration-"+Portfolio_Name),0);
   string t=DoubleToString(GlobalVariableGet("Threshold-"+Portfolio_Name),2);
   string n=DoubleToString(GlobalVariableGet("Increment-"+Portfolio_Name),0);
   string v=DoubleToString(GlobalVariableGet("Volume-"+Portfolio_Name),2);
   string b=DoubleToString(GlobalVariableGet("Breakeven-"+Portfolio_Name),2);
   PlaceLabel("Iteration-"+Portfolio_Name,-100,-120,Text_Corner,i,Text_Color,FONT_TEXT,SIZE_TEXT);
   PlaceLabel("Threshold-"+Portfolio_Name,-100,-120,Text_Corner,t,Text_Color,FONT_TEXT,SIZE_TEXT);
   PlaceLabel("Increment-"+Portfolio_Name,-100,-120,Text_Corner,n,Text_Color,FONT_TEXT,SIZE_TEXT);
   PlaceLabel("Volume-"+Portfolio_Name,-100,-100,Text_Corner,v,Text_Color,FONT_TEXT,SIZE_TEXT);
   PlaceLabel("Breakeven-"+Portfolio_Name,-100,-100,Text_Corner,b,Text_Color,FONT_TEXT,SIZE_TEXT);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
double GetProfit(int number)
  {
   double result=0;
   string filter="";

   if(number!=0)
      filter=Portfolio_Name+"."+IntegerToString(number);
   else
      if(Terminal_Mode)
         filter=Comment_Filter;
      else
         filter=Portfolio_Name;

   for(int n=OrdersTotal(); n>=1; n--)
     {
      if(!OrderSelect(n-1,SELECT_BY_POS,MODE_TRADES))
         continue;
      if(filter!="")
         if(StringFind(OrderComment(),filter,0)==-1)
            continue;
      result+=OrderProfit()+OrderCommission()+OrderSwap();
     }

   return(result);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
double GetVolume(int n)
  {
   if(Volume_Progression==equal)
     {
      return(1);
     }
   else
      if(Volume_Progression==fibo)
        {
         if(n<3)
            return(1);
         int f=1,s=1;
         while(n>2)
           {
            f=f+s*2;
            s=f-s;
            f=f-s;
            n--;
           }
         return(s);
        }
      else
         if(Volume_Progression==martin)
           {
            return(MathPow(2,n-1));
           }
         else
            if(Volume_Progression==linear)
              {
               return(n);
              }
   return(1);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void Report(string text)
  {
   if(Auto_Alerts)
     {
      Alert(text);
      if(Push_Notifications)
         SendNotification(text);
     }
   else
      Print(text);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void Alarm(string text)
  {
   Alert(text);
   if(Push_Notifications)
      SendNotification(text);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CheckLimits()
  {
   if(Stop_Loss!=0)
      if(volume!=0)
         if(profit<=-limit_loss)
           {
            Report(Portfolio_Name+": Stop-loss triggered at "+DoubleToStr(profit,2));
            DoClose(true,0);
            UpdateStatus();
            DoSave(true);
            MakeScreenshot(1);
           }
   if(Take_Profit!=0)
      if(volume!=0)
         if(profit>=limit_profit)
           {
            Report(Portfolio_Name+": Take-profit triggered at "+DoubleToStr(profit,2));
            DoClose(true,0);
            UpdateStatus();
            DoSave(true);
            MakeScreenshot(1);
           }
   if(Breakeven_Level!=0)
      if(breakeven)
         if(volume!=0)
            if(profit<=0)
              {
               Report(Portfolio_Name+": Break-even triggered at "+DoubleToStr(profit,2));
               DoClose(true,0);
               UpdateStatus();
               DoSave(true);
               MakeScreenshot(1);
              }
   if(Breakeven_Level!=0)
      if(!breakeven)
         if(volume!=0)
            if(profit>=limit_breakeven)
              {
               Report(Portfolio_Name+": Breakeven activated at "+DoubleToStr(profit,2));
               GlobalVariableSet("Breakeven-"+Portfolio_Name,1);
               BackupGlobals();
               UpdateStatus();
               DoSave(true);
               MakeScreenshot(1);
              }
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CheckTrailing()
  {

   if(!Use_Trailing)
      return;
   if(volume==0)
      return;

   double trail_long=0;
   double trail_short=0;
   double cond_long=0;
   double cond_short=0;


   switch(Trailing_Long)
     {
      case main:
         trail_long=GlobalVariableGet("Main-"+Portfolio_Name);
         break;
      case fast:
         trail_long=GlobalVariableGet("Fast-"+Portfolio_Name);
         break;
      case slow:
         trail_long=GlobalVariableGet("Slow-"+Portfolio_Name);
         break;
      case upper:
         trail_long=GlobalVariableGet("Upper-"+Portfolio_Name);
         break;
      case lower:
         trail_long=GlobalVariableGet("Lower-"+Portfolio_Name);
         break;
      case model:
         trail_long=GlobalVariableGet("Model-"+Portfolio_Name);
         break;
      case zero:
         trail_long=0;
         break;
     }


   switch(Trailing_Short)
     {
      case main:
         trail_short=GlobalVariableGet("Main-"+Portfolio_Name);
         break;
      case fast:
         trail_short=GlobalVariableGet("Fast-"+Portfolio_Name);
         break;
      case slow:
         trail_short=GlobalVariableGet("Slow-"+Portfolio_Name);
         break;
      case upper:
         trail_short=GlobalVariableGet("Upper-"+Portfolio_Name);
         break;
      case lower:
         trail_short=GlobalVariableGet("Lower-"+Portfolio_Name);
         break;
      case model:
         trail_short=GlobalVariableGet("Model-"+Portfolio_Name);
         break;
      case zero:
         trail_short=0;
         break;
     }


   switch(Condition_Long)
     {
      case main:
         cond_long=GlobalVariableGet("Main-"+Portfolio_Name);
         break;
      case fast:
         cond_long=GlobalVariableGet("Fast-"+Portfolio_Name);
         break;
      case slow:
         cond_long=GlobalVariableGet("Slow-"+Portfolio_Name);
         break;
      case upper:
         cond_long=GlobalVariableGet("Upper-"+Portfolio_Name);
         break;
      case lower:
         cond_long=GlobalVariableGet("Lower-"+Portfolio_Name);
         break;
      case model:
         cond_long=GlobalVariableGet("Model-"+Portfolio_Name);
         break;
      case zero:
         cond_long=0;
         break;
     }


   switch(Condition_Short)
     {
      case main:
         cond_short=GlobalVariableGet("Main-"+Portfolio_Name);
         break;
      case fast:
         cond_short=GlobalVariableGet("Fast-"+Portfolio_Name);
         break;
      case slow:
         cond_short=GlobalVariableGet("Slow-"+Portfolio_Name);
         break;
      case upper:
         cond_short=GlobalVariableGet("Upper-"+Portfolio_Name);
         break;
      case lower:
         cond_short=GlobalVariableGet("Lower-"+Portfolio_Name);
         break;
      case model:
         cond_short=GlobalVariableGet("Model-"+Portfolio_Name);
         break;
      case zero:
         cond_short=0;
         break;
     }


   if(volume>0)
      if(current<=trail_long)
         if(current>=cond_long)
           {
            Report(Portfolio_Name+": Trailing triggered at "+DoubleToStr(profit,2));
            DoClose(true,0);
            UpdateStatus();
            DoSave(true);
            MakeScreenshot(1);
           }

   if(volume<0)
      if(current<=trail_short)
         if(current>=cond_short)
           {
            Report(Portfolio_Name+": Trailing triggered at "+DoubleToStr(profit,2));
            DoClose(true,0);
            UpdateStatus();
            DoSave(true);
            MakeScreenshot(1);
           }

  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CheckIterations()
  {

   if(!Use_Adding && !Use_Correction)
      return;
   if(!IsTradeAllowed())
      return;

   if(current==EMPTY_VALUE)
      return;
   if(previous==EMPTY_VALUE)
      return;
   if(current==0)
      return;
   if(previous==0)
      return;

   datetime now_time=TimeCurrent();


   if(iteration==0)
     {
      iteration=Starting_From;
      while(now_time>=iteration)
         iteration+=Iteration_Minutes*60;
      GlobalVariableSet("Iteration-"+Portfolio_Name,iteration);
      BackupGlobals();
     }


   if(profit==0)
     {
      if(now_time<Starting_From)
         return;
      if(now_time>Finishing_From)
         return;
     }


   int minute_now    =TimeHour(now_time)*60+TimeMinute(now_time);
   int minute_stop   =Stop_Hour*60+Stop_Minute;
   int minute_resume =Resume_Hour*60+Resume_Minute;
   if(minute_now>=minute_stop || minute_now<minute_resume)
      return;


   if(now_time<iteration)
      return;
   while(now_time>=iteration)
      iteration+=Iteration_Minutes*60;
   GlobalVariableSet("Iteration-"+Portfolio_Name,iteration);
   BackupGlobals();


   bool permit_buy=false;
   if(Filter_Op_Buy==above)
      if(current>level_buy)
         permit_buy=true;
   if(Filter_Op_Buy==below)
      if(current<level_buy)
         permit_buy=true;
   if(Filter_Op_Buy==always)
      permit_buy=true;


   bool permit_sell=false;
   if(Filter_Op_Sell==above)
      if(current>level_sell)
         permit_sell=true;
   if(Filter_Op_Sell==below)
      if(current<level_sell)
         permit_sell=true;
   if(Filter_Op_Sell==always)
      permit_sell=true;


   bool permit_reverse=false;
   if(permit_buy)
      if(!permit_sell)
         if(volume<0)
            permit_reverse=true;
   if(permit_sell)
      if(!permit_buy)
         if(volume>0)
            permit_reverse=true;


   if(Filter_Quality!=0)
      if(quality!=0)
         if(quality>Filter_Quality)
           {
            permit_buy=false;
            permit_sell=false;
           }


   if(permit_reverse)
     {
      if(Reversal_Signal==close)
        {
         Report(Portfolio_Name+": Closing at reversal signal...");
         DoClose(true,0);
        }
      if(Reversal_Signal==reverse)
        {
         Report(Portfolio_Name+": Performing positions reversal...");
         DoTrans(-volume,true,0);
        }
     }


   if(Use_Adding)
     {
      bool check1=(profit<=threshold && Adding_Threshold<0);
      bool check2=(profit>=threshold && Adding_Threshold>0);
      bool check3=(Adding_Threshold==0);
      if((check1 || check2 || check3) && (permit_buy || permit_sell))
        {
         double size=GetVolume(increment+1);
         Report(Portfolio_Name+": Adding iteration #"+string(increment+1));
         bool success;
         if(permit_buy)
            if(!permit_sell)
               success=DoOpen(+size,true,increment+1);
         if(permit_sell)
            if(!permit_buy)
               success=DoOpen(-size,true,increment+1);
         if(success)
           {
            UpdateStatus();
            DoSave(true);
           }
        }
     }


   if(Use_Correction)
      for(int m=1; m<=increment; m++)
        {
         double subprofit=GetProfit(m);
         double subvolume=GetVolume(m);
         if(Correction_Loss!=0)
            if(subprofit<=-Correction_Loss*MathAbs(subvolume))
              {
               Report(Portfolio_Name+": Correcting iteration #"+string(m));
               bool success=DoTrans(subvolume,true,m);
               if(success)
                 {
                  UpdateStatus();
                  DoSave(true);
                 }
              }
         if(Correction_Profit!=0)
            if(subprofit>=Correction_Profit*MathAbs(subvolume))
              {
               Report(Portfolio_Name+": Fixing iteration #"+string(m));
               bool success=DoClose(true,m);
               if(success)
                 {
                  UpdateStatus();
                  DoSave(true);
                 }
              }
        }

   MakeScreenshot(2);

  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CheckLines()
  {

   if(!Use_Lines)
      return;
   if(current==EMPTY_VALUE)
      return;
   if(previous==EMPTY_VALUE)
      return;
   if(current==0)
      return;
   if(previous==0)
      return;

   datetime now_time=TimeCurrent();
   if(profit==0)
      if(now_time<Starting_From)
         return;
   if(profit==0)
      if(now_time>Finishing_From)
         return;

   int minute_now    =TimeHour(now_time)*60+TimeMinute(now_time);
   int minute_stop   =Stop_Hour*60+Stop_Minute;
   int minute_resume =Resume_Hour*60+Resume_Minute;
   if(minute_now>=minute_stop || minute_now<minute_resume)
      return;

   for(int n=ObjectsTotal(); n>=1; n--)
     {
      string name=ObjectName(n-1);
      if(ObjectFind(name)!=window)
         continue;
      int object_type=ObjectType(name);
      if(object_type!=OBJ_TREND && object_type!=OBJ_HLINE)
         continue;

      string         text=ObjectDescription(name);
      bool       type_buy=(StringFind(text,"BUY")!=-1);
      bool      type_sell=(StringFind(text,"SELL")!=-1);
      bool     type_close=(StringFind(text,"CLOSE")!=-1);
      bool    type_upward=(StringFind(text,"UPWARD")!=-1);
      bool  type_downward=(StringFind(text,"DOWNWARD")!=-1);
      bool       type_fix=(StringFind(text,"FIX")!=-1);
      bool     type_alert=(StringFind(text,"ALERT")!=-1);
      bool   type_disable=(StringFind(text,"DISABLE")!=-1);

      if(!type_buy && !type_sell && !type_close && !type_fix &&
         !type_upward && !type_downward && !type_alert && !type_disable)
         continue;

      double level=0;
      if(object_type==OBJ_TREND)
         level=NormalizeDouble(ObjectGetValueByShift(name,0),2);
      if(object_type==OBJ_HLINE)
         level=NormalizeDouble(ObjectGet(name,OBJPROP_PRICE1),2);

      bool cross_up = (previous<level && current>=level);
      bool cross_dn = (previous>level && current<=level);
      bool crossing = cross_dn || cross_up;
      if(!crossing)
         continue;

      double modifier=1;
      int length=StringLen(text);
      int index=StringFind(text,":");
      if(index!=-1)
         modifier=StrToDouble(StringSubstr(text,index+1,length-index-1));

      int counter=INT_MAX;
      int left=StringFind(text,"{");
      int right=StringFind(text,"}");
      if(left!=-1 && right!=-1)
         counter=StrToInteger(StringSubstr(text,left+1,right-1));


      if(type_buy)
         if(current<=level+Gap_Protect)
           {
            Report(Portfolio_Name+": BUY level triggered...");
            DoOpen(modifier,true,0);
            ObjectSetText(name,"LONG:"+string(modifier));
            UpdateStatus();
            DoSave(true);
           }


      if(type_sell)
         if(current>=level-Gap_Protect)
           {
            Report(Portfolio_Name+": SELL level triggered...");
            DoOpen(-modifier,true,0);
            ObjectSetText(name,"SHORT:"+string(modifier));
            UpdateStatus();
            DoSave(true);
           }


      if(type_upward)
        {
         if(counter>0)
            if(volume==0)
               if(current<=level+Gap_Protect)
                 {
                  Report(Portfolio_Name+": UPWARD first opening...");
                  DoOpen(1,true,0);
                  if(counter!=INT_MAX)
                     ObjectSetText(name,"{"+string(counter-1)+"}"+"UPWARD:"+string(modifier));
                  UpdateStatus();
                  DoSave(true);
                 }
         if(counter>0)
            if(volume<0)
              {
               Report(Portfolio_Name+": UPWARD reversal triggered...");
               DoTrans(-volume*modifier,true,0);
               if(counter!=INT_MAX)
                  ObjectSetText(name,"{"+string(counter-1)+"}"+"UPWARD:"+string(modifier));
               UpdateStatus();
               DoSave(true);
              }
         if(counter<=0)
            if(volume<0)
              {
               Report(Portfolio_Name+": UPWARD limit reached...");
               DoClose(true,0);
               ObjectSetText(name,"UP:"+string(modifier));
               UpdateStatus();
               DoSave(true);
              }
        }


      if(type_downward)
        {
         if(counter>0)
            if(volume==0)
               if(current>=level-Gap_Protect)
                 {
                  Report(Portfolio_Name+": DOWNWARD first opening...");
                  DoOpen(-1,true,0);
                  if(counter!=INT_MAX)
                     ObjectSetText(name,"{"+string(counter-1)+"}"+"DOWNWARD:"+string(modifier));
                  UpdateStatus();
                  DoSave(true);
                 }
         if(counter>0)
            if(volume>0)
              {
               Report(Portfolio_Name+": DOWNWARD reversal triggered...");
               DoTrans(-volume*modifier,true,0);
               if(counter!=INT_MAX)
                  ObjectSetText(name,"{"+string(counter-1)+"}"+"DOWNWARD:"+string(modifier));
               UpdateStatus();
               DoSave(true);
              }
         if(counter<=0)
            if(volume>0)
              {
               Report(Portfolio_Name+": DOWNWARD limit reached...");
               DoClose(true,0);
               ObjectSetText(name,"DOWN:"+string(modifier));
               UpdateStatus();
               DoSave(true);
              }
        }


      if(type_fix)
         if(profit!=0)
            if(volume!=0)
              {
               Report(Portfolio_Name+": FIX level triggered with P/L:"+DoubleToStr(profit,2));
               DoClose(true,0);
               UpdateStatus();
               DoSave(true);
              }


      if(type_close)
         if(profit!=0)
            if(volume!=0)
              {
               Report(Portfolio_Name+": CLOSE level triggered with P/L:"+DoubleToStr(profit,2));
               if(volume==0)
                  DoClose(true,0);
               else
                  DoTrans(volume*(1-modifier),true,0);
               ObjectSetText(name,"EXIT");
               UpdateStatus();
               DoSave(true);
              }


      if(type_alert)
        {
         Alarm(Portfolio_Name+": "+DoubleToStr(level,2)+" level reached...");
         ObjectSetText(name,"LEVEL");
         UpdateStatus();
        }


      if(type_disable)
        {
         Report(Portfolio_Name+": All lines disabled...");
         DisableAllLines();
         UpdateStatus();
        }
        
     }

   MakeScreenshot(3);

  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void DisableAllLines()
  {
   for(int n=ObjectsTotal(); n>=1; n--)
     {
      string name=ObjectName(n-1);
      if(ObjectFind(name)!=window)
         continue;
      if(ObjectType(name)!=OBJ_TREND && ObjectType(name)!=OBJ_HLINE)
         continue;
      string text=ObjectDescription(name);

      bool       type_buy=(StringFind(text,"BUY")!=-1);
      bool      type_sell=(StringFind(text,"SELL")!=-1);
      bool     type_close=(StringFind(text,"CLOSE")!=-1);
      bool    type_upward=(StringFind(text,"UPWARD")!=-1);
      bool  type_downward=(StringFind(text,"DOWNWARD")!=-1);
      bool       type_fix=(StringFind(text,"FIX")!=-1);
      bool     type_alert=(StringFind(text,"ALERT")!=-1);
      bool   type_disable=(StringFind(text,"DISABLE")!=-1);

      double modifier=1;
      int length=StringLen(text);
      int index=StringFind(text,":");
      if(index!=-1)
         modifier=StringToDouble(StringSubstr(text,index+1,length-index-1));

      if(type_buy)
         ObjectSetText(name,"LONG:"+string(modifier));
      if(type_sell)
         ObjectSetText(name,"SHORT:"+string(modifier));
      if(type_close)
         ObjectSetText(name,"EXIT");
      if(type_fix)
         ObjectSetText(name,"QUIT");
      if(type_upward)
         ObjectSetText(name,"UP:"+string(modifier));
      if(type_downward)
         ObjectSetText(name,"DOWN:"+string(modifier));
      if(type_disable)
         ObjectSetText(name,"OFF");
     }
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool DoTrans(double factor,bool automatic,int number)
  {

   ReadScreenFormula();

   for(int i=0; i<total; i++)
      if(!IsTradeAllowed(SYMBOLS[i],TimeCurrent()))
        {
         Alarm(Portfolio_Name+": Out of trading sessions! - "+SYMBOLS[i]);
         return(false);
        }

   for(int i=0; i<total; i++)
      if(LOTS[i]!=0)
         if(MathAbs(LOTS[i])<MarketInfo(SYMBOLS[i],MODE_MINLOT))
           {
            Alarm(Portfolio_Name+": Minimum trading lot violated for symbol: "+SYMBOLS[i]);
            return(false);
           }

   if(Max_Volume>0)
      if(MathAbs(factor)>Max_Volume)
        {
         Alarm(Portfolio_Name+": Maximum volume exceeded!");
         return(false);
        }

   if(!automatic && Manual_Confirm)
     {
      string text="Transforming portfolio: "+Portfolio_Name;
      if(MessageBox(text,"",MB_OKCANCEL|MB_ICONINFORMATION)==IDCANCEL)
        {
         Print("Cancelled!");
         return(false);
        }
      else
         Print("Confirmed!");
     }


   int magic=Magic_Number;
   string comment=Portfolio_Name;
   if(number>0)
      comment+="."+IntegerToString(number);


   string filter="";
   if(number!=0)
      filter=Portfolio_Name+"."+IntegerToString(number);
   else
      if(Terminal_Mode)
         filter=Comment_Filter;
      else
         filter=Portfolio_Name;


   for(int i=0; i<total; i++)
     {

      double sum_lots=0;
      for(int n=OrdersTotal(); n>=1; n--)
        {
         if(!OrderSelect(n-1,SELECT_BY_POS,MODE_TRADES))
            continue;
         if(OrderSymbol()!=SYMBOLS[i])
            continue;
         if(filter!="")
            if(StringFind(OrderComment(),filter,0)==-1)
               continue;
         if(OrderType()==OP_BUY)
            sum_lots+=OrderLots();
         if(OrderType()==OP_SELL)
            sum_lots-=OrderLots();
        }


      double new_lot=NormalizeDouble(factor*LOTS[i],Lots_Digits);
      double delta=NormalizeDouble(new_lot-sum_lots,Lots_Digits);
      if(delta==0)
         continue;


      for(int n=OrdersTotal(); n>=1; n--)
        {

         if(!OrderSelect(n-1,SELECT_BY_POS,MODE_TRADES))
            continue;
         if(OrderSymbol()!=SYMBOLS[i])
            continue;
         if(filter!="")
            if(StringFind(OrderComment(),filter,0)==-1)
               continue;
         if(new_lot!=0)
            if(OrderType()==OP_BUY && delta>0)
               continue;
         if(new_lot!=0)
            if(OrderType()==OP_SELL && delta<0)
               continue;
         if(new_lot!=0)
            if(OrderLots()>MathAbs(delta))
               continue;


         for(int k=Retry_Times; k>0; k--)
           {
            bool check=false;
            Print(Portfolio_Name,": Sending CLOSE ",SYMBOLS[i]," ",DoubleToStr(OrderLots(),Lots_Digits));
            if(OrderType()==OP_BUY)
               check=OrderClose(OrderTicket(),OrderLots(),MarketInfo(SYMBOLS[i],MODE_BID),0);
            if(OrderType()==OP_SELL)
               check=OrderClose(OrderTicket(),OrderLots(),MarketInfo(SYMBOLS[i],MODE_ASK),0);
            if(check)
               break;


            string message=Portfolio_Name+": Trading error! - "+ErrorDescription(GetLastError())+" for "+SYMBOLS[i];
            Print(message);
            if(automatic)
               Sleep(Retry_Delay);
            else
               if(MessageBox(message,"",MB_RETRYCANCEL|MB_ICONERROR)==IDCANCEL)
                  return(false);


            if(IsStopped())
               break;
            if(k>1)
               continue;


            Alarm(Portfolio_Name+": Failed to close! - "+ErrorDescription(GetLastError())+" for "+SYMBOLS[i]);
            return(false);
           }


         if(delta>0)
            delta=NormalizeDouble(delta-OrderLots(),Lots_Digits);
         if(delta<0)
            delta=NormalizeDouble(delta+OrderLots(),Lots_Digits);
        }


      if(delta!=0)
         for(int k=Retry_Times; k>0; k--)
           {
            long ticket=-1;
            if(delta>0)
               Print(Portfolio_Name,": Sending BUY ",SYMBOLS[i]," ",DoubleToStr(MathAbs(delta),Lots_Digits));
            if(delta<0)
               Print(Portfolio_Name,": Sending SELL ",SYMBOLS[i]," ",DoubleToStr(MathAbs(delta),Lots_Digits));
            if(delta>0)
               ticket=OrderSend(SYMBOLS[i],OP_BUY,MathAbs(delta),MarketInfo(SYMBOLS[i],MODE_ASK),0,0,0,comment,magic);
            if(delta<0)
               ticket=OrderSend(SYMBOLS[i],OP_SELL,MathAbs(delta),MarketInfo(SYMBOLS[i],MODE_BID),0,0,0,comment,magic);
            if(ticket!=-1)
               break;

            string message=Portfolio_Name+": Trading error! - "+ErrorDescription(GetLastError())+" for "+SYMBOLS[i];
            Print(message);
            if(automatic)
               Sleep(Retry_Delay);
            else
               if(MessageBox(message,"",MB_RETRYCANCEL|MB_ICONERROR)==IDCANCEL)
                  return(false);

            if(IsStopped())
               break;
            if(k>1)
               continue;
            Alarm(Portfolio_Name+": Failed to open! - "+ErrorDescription(GetLastError())+" for "+SYMBOLS[i]);
            return(false);
           }
     }


   for(int n=OrdersTotal(); n>=1; n--)
     {
      if(!OrderSelect(n-1,SELECT_BY_POS,MODE_TRADES))
         continue;
      if(filter!="")
         if(StringFind(OrderComment(),filter,0)==-1)
            continue;

      bool presence=false;
      for(int i=0; i<total; i++)
         if(SYMBOLS[i]==OrderSymbol())
            presence=true;
      if(presence)
         continue;

      for(int k=Retry_Times; k>0; k--)
        {
         bool check=false;
         Print(Portfolio_Name,": Sending CLOSE ",OrderSymbol()," ",DoubleToStr(OrderLots(),Lots_Digits));
         if(OrderType()==OP_BUY)
            check=OrderClose(OrderTicket(),OrderLots(),MarketInfo(OrderSymbol(),MODE_BID),0);
         if(OrderType()==OP_SELL)
            check=OrderClose(OrderTicket(),OrderLots(),MarketInfo(OrderSymbol(),MODE_ASK),0);
         if(check)
            break;


         string message=Portfolio_Name+": Trading error! - "+ErrorDescription(GetLastError())+" for "+OrderSymbol();
         Print(message);
         if(automatic)
            Sleep(Retry_Delay);
         else
            if(MessageBox(message,"",MB_RETRYCANCEL|MB_ICONERROR)==IDCANCEL)
               return(false);


         if(IsStopped())
            break;
         if(k>1)
            continue;

         Alarm(Portfolio_Name+": Failed to close! - "+ErrorDescription(GetLastError())+" for "+OrderSymbol());
         return(false);
        }
     }

   volume=factor;
   GlobalVariableSet("Volume-"+Portfolio_Name,volume);
   BackupGlobals();

   if(Show_Trades)
     {
      datetime t=TimeCurrent();
      PlaceVertical("Trade-transformation-"+string(t),t,LINES_COLOR,STYLE_DASHDOTDOT);
     }


   current=GlobalVariableGet("Portfolio-"+Portfolio_Name);
   string text="Portfolio "+Portfolio_Name+" transformed at "+DoubleToStr(current,2)+" volume: "+string(volume);
   if(!automatic && Manual_Confirm)
      MessageBox(text,"");
   else
      Print(text);
   return(true);

  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool DoOpen(double position,bool automatic,int number)
  {

   if(position==0)
     {
      Alarm(Portfolio_Name+": Zero volume requested!");
      return(false);
     }

   ReadScreenFormula();

   for(int i=0; i<total; i++)
      if(!IsTradeAllowed(SYMBOLS[i],TimeCurrent()))
        {
         Alarm(Portfolio_Name+": Out of trading sessions! - "+SYMBOLS[i]);
         return(false);
        }

   for(int i=0; i<total; i++)
      if(LOTS[i]!=0)
         if(MathAbs(LOTS[i])<MarketInfo(SYMBOLS[i],MODE_MINLOT))
           {
            Alarm(Portfolio_Name+": Minimum trading lot violated for symbol: "+SYMBOLS[i]);
            return(false);
           }

   volume=GlobalVariableGet("Volume-"+Portfolio_Name);

   if(Max_Volume>0)
      if(MathAbs(position+volume)>Max_Volume)
        {
         Alarm(Portfolio_Name+": Maximum portfolio volume exceeded!");
         return(false);
        }

   if(!automatic && Manual_Confirm)
     {
      string text;
      if(position>0)
         text="Opening LONG portfolio: "+Portfolio_Name;
      if(position<0)
         text="Opening SHORT portfolio: "+Portfolio_Name;
      if(MessageBox(text,"",MB_OKCANCEL|MB_ICONINFORMATION)==IDCANCEL)
        {
         Print("Cancelled!");
         return(false);
        }
      else
         Print("Confirmed!");
     }

   int magic=Magic_Number;
   string comment=Portfolio_Name;
   if(number>0)
      comment+="."+IntegerToString(number);

   for(int i=0; i<total; i++)
      for(int k=Retry_Times; k>0; k--)
        {
         long ticket=-1;
         double lot=NormalizeDouble(LOTS[i]*position,Lots_Digits);

         if(lot==0)
            break;
         if(lot>0)
            Print(Portfolio_Name,": Sending BUY ",SYMBOLS[i]," ",DoubleToStr(MathAbs(lot),Lots_Digits));
         if(lot<0)
            Print(Portfolio_Name,": Sending SELL ",SYMBOLS[i]," ",DoubleToStr(MathAbs(lot),Lots_Digits));
         if(lot>0)
            ticket=OrderSend(SYMBOLS[i],OP_BUY,MathAbs(lot),MarketInfo(SYMBOLS[i],MODE_ASK),0,0,0,comment,magic);
         if(lot<0)
            ticket=OrderSend(SYMBOLS[i],OP_SELL,MathAbs(lot),MarketInfo(SYMBOLS[i],MODE_BID),0,0,0,comment,magic);
         if(ticket!=-1)
            break;

         string message=Portfolio_Name+": Trading error! - "+ErrorDescription(GetLastError())+" for "+SYMBOLS[i];
         Print(message);
         
         if(automatic)
            Sleep(Retry_Delay);
         else
            if(MessageBox(message,"",MB_RETRYCANCEL|MB_ICONERROR)==IDCANCEL)
               return(false);

         if(IsStopped())
            break;
         if(k>1)
            continue;

         Alarm(Portfolio_Name+": Failed to open! - "+ErrorDescription(GetLastError())+" for "+SYMBOLS[i]);
         return(false);
        }


   if(number==0)
      volume+=position;
   else
      volume=(MathAbs(volume)+MathAbs(position))*((position>0)?1:-1);

   increment++;
   threshold+=Adding_Threshold*MathAbs(volume);
   GlobalVariableSet("Volume-"+Portfolio_Name,volume);
   GlobalVariableSet("Increment-"+Portfolio_Name,increment);
   GlobalVariableSet("Threshold-"+Portfolio_Name,threshold);
   BackupGlobals();


   if(Show_Trades)
     {
      datetime t=TimeCurrent();
      PlaceVertical("Trade-opening-"+string(t),t,LINES_COLOR,STYLE_DASHDOTDOT);
     }


   string text;
   current=GlobalVariableGet("Portfolio-"+Portfolio_Name);

   if(position>0)
      text="LONG portfolio " +Portfolio_Name+" opened at "+DoubleToStr(current,2)+" volume: "+string(volume);
   if(position<0)
      text="SHORT portfolio "+Portfolio_Name+" opened at "+DoubleToStr(current,2)+" volume: "+string(volume);

   if(!automatic && Manual_Confirm)
      MessageBox(text,"");
   else
      Print(text);

   return(true);

  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool DoClose(bool automatic,int number)
  {

   ReadScreenFormula();

   for(int i=0; i<total; i++)
      if(!IsTradeAllowed(SYMBOLS[i],TimeCurrent()))
        {
         Alarm(Portfolio_Name+": Out of trading sessions! - "+SYMBOLS[i]);
         return(false);
        }

   if(!automatic && Manual_Confirm)
     {
      string text="Closing out portfolio: "+Portfolio_Name;
      if(MessageBox(text,"",MB_OKCANCEL|MB_ICONINFORMATION)==IDCANCEL)
        {
         Print("Cancelled!");
         return(false);
        }
      else
         Print("Confirmed!");
     }


   string comment=Portfolio_Name;
   if(number>0)
      comment+="."+IntegerToString(number);
   int count=0;


   string filter="";
   if(number!=0)
      filter=Portfolio_Name+"."+IntegerToString(number);
   else
      if(Terminal_Mode)
         filter=Comment_Filter;
      else
         filter=Portfolio_Name;


   for(int n=OrdersTotal(); n>=1; n--)
     {
      if(!OrderSelect(n-1,SELECT_BY_POS,MODE_TRADES))
         continue;
      if(filter!="")
         if(StringFind(OrderComment(),filter,0)==-1)
            continue;
      count++;

      for(int k=Retry_Times; k>0; k--)
        {
         bool check=false;
         Print(Portfolio_Name,": Sending CLOSE ",OrderSymbol()," ",DoubleToStr(OrderLots(),Lots_Digits));
         if(OrderType()==OP_BUY)
            check=OrderClose(OrderTicket(),OrderLots(),MarketInfo(OrderSymbol(),MODE_BID),0);
         if(OrderType()==OP_SELL)
            check=OrderClose(OrderTicket(),OrderLots(),MarketInfo(OrderSymbol(),MODE_ASK),0);
         if(check)
            break;


         string message=Portfolio_Name+": Trading error! - "+ErrorDescription(GetLastError())+" for "+OrderSymbol();
         Print(message);
         if(automatic)
            Sleep(Retry_Delay);
         else
            if(MessageBox(message,"",MB_RETRYCANCEL|MB_ICONERROR)==IDCANCEL)
               return(false);


         if(IsStopped())
            break;
         if(k>1)
            continue;

         Alarm(Portfolio_Name+": Failed to close! - "+ErrorDescription(GetLastError())+" for "+OrderSymbol());
         return(false);
        }
     }


   if(number==0)
     {
      volume=0;
      increment=0;
      threshold=0;
      iteration=0;
      breakeven=false;
      ObjectDelete("Zero-"+Portfolio_Name);
      ObjectDelete("Stop-"+Portfolio_Name);
      ObjectDelete("Take-"+Portfolio_Name);
      ObjectDelete("Safe-"+Portfolio_Name);
      GlobalVariableDel("Volume-"+Portfolio_Name);
      GlobalVariableDel("Increment-"+Portfolio_Name);
      GlobalVariableDel("Threshold-"+Portfolio_Name);
      GlobalVariableDel("Iteration-"+Portfolio_Name);
      GlobalVariableDel("Breakeven-"+Portfolio_Name);
      ObjectDelete("Volume-"+Portfolio_Name);
      ObjectDelete("Increment-"+Portfolio_Name);
      ObjectDelete("Threshold-"+Portfolio_Name);
      ObjectDelete("Iteration-"+Portfolio_Name);
      ObjectDelete("Breakeven-"+Portfolio_Name);
     }
   else
     {
      threshold-=Adding_Threshold*MathAbs(volume);
      if(volume>0)
         volume-=GetVolume(number);
      if(volume<0)
         volume+=GetVolume(number);
      increment--;
      GlobalVariableSet("Volume-"+Portfolio_Name,volume);
      GlobalVariableSet("Increment-"+Portfolio_Name,increment);
      GlobalVariableSet("Threshold-"+Portfolio_Name,threshold);
      BackupGlobals();
     }


   if(Show_Trades)
     {
      datetime t=TimeCurrent();
      if(number==0)
         PlaceVertical("Trade-closing-"+string(t),t,LINES_COLOR,STYLE_DASHDOTDOT);
      else
         PlaceVertical("Trade-part-closing-"+string(t),t,LINES_COLOR,STYLE_DASHDOTDOT);
     }


   string text;
   current=GlobalVariableGet("Portfolio-"+Portfolio_Name);

   if(count>0)
      text="Portfolio "+Portfolio_Name+" closed out at "+DoubleToStr(current,2);
   else
      text="No positions for portfolio: "+Portfolio_Name;

   if(!automatic && Manual_Confirm)
      MessageBox(text,"");
   else
      Print(text);

   return(true);

  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void ReadScreenFormula()
  {
   total=0;
   string parts[];
   for(int i=0; i<ObjectsTotal(); i++)
     {
      string name=ObjectName(i);
      if(StringFind(name,"Formula-label-"+Portfolio_Name+"-")==-1)
         continue;
      string text=ObjectGetString(chart,name,OBJPROP_TEXT,0);
      int num=StringSplit(text,SPECIAL,parts);
      if(num==2)
        {
         SYMBOLS[total]=parts[0];
         LOTS[total]=StrToDouble(parts[1]);
        }
      else
         if(num==1)
           {
            SYMBOLS[i]=parts[0];
            LOTS[total]=1;
           }
         else
           {
            Alarm("Incorrect portfolio formula!");
           }
      total++;
     }
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void DoSave(bool auto)
  {
   if(!Template_Autosave && auto)
      return;
   bool ok=ChartSaveTemplate(chart,Portfolio_Name);
   string text=ok?("Saved to template: "+Portfolio_Name):("Template saving failed!");
   if(auto)
      Print(text);
   else
      MessageBox(text,"");
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void MakeScreenshot(int event)
  {
   static const string SCREEN_EVENT[4]= {" TIMER"," LIMITS"," SERIES"," LINES" };
   if(Screenshot_Seconds<=0)
      return;
   datetime now_time=TimeCurrent();
   if(now_time>screen_timer || event>0)
     {
      if(screen_timer==0)
         screen_timer=Starting_From;
      if(event==0)
         while(screen_timer<=now_time)
            screen_timer+=Screenshot_Seconds;
      string time_text=TimeToStr(now_time,TIME_DATE|TIME_SECONDS);
      StringReplace(time_text,".","-");
      StringReplace(time_text,":","-");
      string name=Portfolio_Name+"\\"+Portfolio_Name+"-"+time_text+SCREEN_EVENT[event]+".gif";
      WindowScreenShot(name,Screenshot_Width,Screenshot_Height,-1,-1);
     }
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void PlaceLabel(string name,int x,int y,int corner,string text,int colour,string font,int size)
  {
   ObjectCreate(name,OBJ_LABEL,window,0,0);
   ObjectSet(name,OBJPROP_CORNER,corner);
   ObjectSet(name,OBJPROP_XDISTANCE,x);
   ObjectSet(name,OBJPROP_YDISTANCE,y);
   ObjectSetText(name,text,size,font,colour);
   ObjectSet(name,OBJPROP_SELECTABLE,false);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void PlaceVertical(string name,datetime time,int colour,int style)
  {
   ObjectCreate(0,name,OBJ_VLINE,window,time,0);
   ObjectSetInteger(0,name,OBJPROP_TIME,time);
   ObjectSetInteger(0,name,OBJPROP_COLOR,colour);
   ObjectSetInteger(0,name,OBJPROP_STYLE,style);
   ObjectSetInteger(0,name,OBJPROP_BACK,true);
   ObjectSetInteger(0,name,OBJPROP_SELECTABLE,false);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void PlaceHorizontal(string name,double price,int colour,int style)
  {
   ObjectCreate(0,name,OBJ_HLINE,window,0,price);
   ObjectSetDouble(0,name,OBJPROP_PRICE,price);
   ObjectSetInteger(0,name,OBJPROP_COLOR,colour);
   ObjectSetInteger(0,name,OBJPROP_STYLE,style);
   ObjectSetInteger(0,name,OBJPROP_BACK,true);
   ObjectSetInteger(0,name,OBJPROP_SELECTABLE,false);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
/*
string ErrorDescription(int code)
  {
   if(code==ERR_NO_ERROR)
      return("NO_ERROR");
   if(code==ERR_NO_RESULT)
      return("NO_RESULT");
   if(code==ERR_COMMON_ERROR)
      return("COMMON_ERROR");
   if(code==ERR_INVALID_TRADE_PARAMETERS)
      return("INVALID_TRADE_PARAMETERS");
   if(code==ERR_SERVER_BUSY)
      return("SERVER_BUSY");
   if(code==ERR_OLD_VERSION)
      return("OLD_VERSION");
   if(code==ERR_NO_CONNECTION)
      return("NO_CONNECTION");
   if(code==ERR_NOT_ENOUGH_RIGHTS)
      return("NOT_ENOUGH_RIGHTS");
   if(code==ERR_TOO_FREQUENT_REQUESTS)
      return("TOO_FREQUENT_REQUESTS");
   if(code==ERR_MALFUNCTIONAL_TRADE)
      return("MALFUNCTIONAL_TRADE");
   if(code==ERR_ACCOUNT_DISABLED)
      return("ACCOUNT_DISABLED");
   if(code==ERR_INVALID_ACCOUNT)
      return("INVALID_ACCOUNT");
   if(code==ERR_TRADE_TIMEOUT)
      return("TRADE_TIMEOUT");
   if(code==ERR_INVALID_PRICE)
      return("INVALID_PRICE");
   if(code==ERR_INVALID_STOPS)
      return("INVALID_STOPS");
   if(code==ERR_INVALID_TRADE_VOLUME)
      return("INVALID_TRADE_VOLUME");
   if(code==ERR_MARKET_CLOSED)
      return("MARKET_CLOSED");
   if(code==ERR_TRADE_DISABLED)
      return("TRADE_DISABLED");
   if(code==ERR_NOT_ENOUGH_MONEY)
      return("NOT_ENOUGH_MONEY");
   if(code==ERR_PRICE_CHANGED)
      return("PRICE_CHANGED");
   if(code==ERR_OFF_QUOTES)
      return("OFF_QUOTES");
   if(code==ERR_BROKER_BUSY)
      return("BROKER_BUSY");
   if(code==ERR_REQUOTE)
      return("REQUOTE");
   if(code==ERR_ORDER_LOCKED)
      return("ORDER_LOCKED");
   if(code==ERR_LONG_POSITIONS_ONLY_ALLOWED)
      return("LONG_POSITIONS_ONLY_ALLOWED");
   if(code==ERR_TOO_MANY_REQUESTS)
      return("TOO_MANY_REQUESTS");
   if(code==ERR_TRADE_MODIFY_DENIED)
      return("TRADE_MODIFY_DENIED");
   if(code==ERR_TRADE_CONTEXT_BUSY)
      return("TRADE_CONTEXT_BUSY");
   if(code==ERR_TRADE_EXPIRATION_DENIED)
      return("TRADE_EXPIRATION_DENIED");
   if(code==ERR_TRADE_TOO_MANY_ORDERS)
      return("TRADE_TOO_MANY_ORDERS");
   if(code==ERR_TRADE_HEDGE_PROHIBITED)
      return("TRADE_HEDGE_PROHIBITED");
   if(code==ERR_TRADE_PROHIBITED_BY_FIFO)
      return("TRADE_PROHIBITED_BY_FIFO");
   if(code==ERR_NO_MQLERROR)
      return("NO_MQLERROR");
   if(code==ERR_WRONG_FUNCTION_POINTER)
      return("WRONG_FUNCTION_POINTER");
   if(code==ERR_ARRAY_INDEX_OUT_OF_RANGE)
      return("ARRAY_INDEX_OUT_OF_RANGE");
   if(code==ERR_NO_MEMORY_FOR_CALL_STACK)
      return("NO_MEMORY_FOR_CALL_STACK");
   if(code==ERR_RECURSIVE_STACK_OVERFLOW)
      return("RECURSIVE_STACK_OVERFLOW");
   if(code==ERR_NOT_ENOUGH_STACK_FOR_PARAM)
      return("NOT_ENOUGH_STACK_FOR_PARAM");
   if(code==ERR_NO_MEMORY_FOR_PARAM_STRING)
      return("NO_MEMORY_FOR_PARAM_STRING");
   if(code==ERR_NO_MEMORY_FOR_TEMP_STRING)
      return("NO_MEMORY_FOR_TEMP_STRING");
   if(code==ERR_NOT_INITIALIZED_STRING)
      return("NOT_INITIALIZED_STRING");
   if(code==ERR_NOT_INITIALIZED_ARRAYSTRING)
      return("NOT_INITIALIZED_ARRAYSTRING");
   if(code==ERR_NO_MEMORY_FOR_ARRAYSTRING)
      return("NO_MEMORY_FOR_ARRAYSTRING");
   if(code==ERR_TOO_LONG_STRING)
      return("TOO_LONG_STRING");
   if(code==ERR_REMAINDER_FROM_ZERO_DIVIDE)
      return("REMAINDER_FROM_ZERO_DIVIDE");
   if(code==ERR_ZERO_DIVIDE)
      return("ZERO_DIVIDE");
   if(code==ERR_UNKNOWN_COMMAND)
      return("UNKNOWN_COMMAND");
   if(code==ERR_WRONG_JUMP)
      return("WRONG_JUMP");
   if(code==ERR_NOT_INITIALIZED_ARRAY)
      return("NOT_INITIALIZED_ARRAY");
   if(code==ERR_DLL_CALLS_NOT_ALLOWED)
      return("DLL_CALLS_NOT_ALLOWED");
   if(code==ERR_CANNOT_LOAD_LIBRARY)
      return("CANNOT_LOAD_LIBRARY");
   if(code==ERR_CANNOT_CALL_FUNCTION)
      return("CANNOT_CALL_FUNCTION");
   if(code==ERR_EXTERNAL_CALLS_NOT_ALLOWED)
      return("EXTERNAL_CALLS_NOT_ALLOWED");
   if(code==ERR_NO_MEMORY_FOR_RETURNED_STR)
      return("NO_MEMORY_FOR_RETURNED_STR");
   if(code==ERR_SYSTEM_BUSY)
      return("SYSTEM_BUSY");
   if(code==ERR_DLLFUNC_CRITICALERROR)
      return("DLLFUNC_CRITICALERROR");
   if(code==ERR_INTERNAL_ERROR)
      return("INTERNAL_ERROR");
   if(code==ERR_OUT_OF_MEMORY)
      return("OUT_OF_MEMORY");
   if(code==ERR_INVALID_POINTER)
      return("INVALID_POINTER");
   if(code==ERR_FORMAT_TOO_MANY_FORMATTERS)
      return("FORMAT_TOO_MANY_FORMATTERS");
   if(code==ERR_FORMAT_TOO_MANY_PARAMETERS)
      return("FORMAT_TOO_MANY_PARAMETERS");
   if(code==ERR_ARRAY_INVALID)
      return("ARRAY_INVALID");
   if(code==ERR_CHART_NOREPLY)
      return("CHART_NOREPLY");
   if(code==ERR_INVALID_FUNCTION_PARAMSCNT)
      return("INVALID_FUNCTION_PARAMSCNT");
   if(code==ERR_INVALID_FUNCTION_PARAMVALUE)
      return("INVALID_FUNCTION_PARAMVALUE");
   if(code==ERR_STRING_FUNCTION_INTERNAL)
      return("STRING_FUNCTION_INTERNAL");
   if(code==ERR_SOME_ARRAY_ERROR)
      return("SOME_ARRAY_ERROR");
   if(code==ERR_INCORRECT_SERIESARRAY_USING)
      return("INCORRECT_SERIESARRAY_USING");
   if(code==ERR_CUSTOM_INDICATOR_ERROR)
      return("CUSTOM_INDICATOR_ERROR");
   if(code==ERR_INCOMPATIBLE_ARRAYS)
      return("INCOMPATIBLE_ARRAYS");
   if(code==ERR_GLOBAL_VARIABLES_PROCESSING)
      return("GLOBAL_VARIABLES_PROCESSING");
   if(code==ERR_GLOBAL_VARIABLE_NOT_FOUND)
      return("GLOBAL_VARIABLE_NOT_FOUND");
   if(code==ERR_FUNC_NOT_ALLOWED_IN_TESTING)
      return("FUNC_NOT_ALLOWED_IN_TESTING");
   if(code==ERR_FUNCTION_NOT_CONFIRMED)
      return("FUNCTION_NOT_CONFIRMED");
   if(code==ERR_SEND_MAIL_ERROR)
      return("SEND_MAIL_ERROR");
   if(code==ERR_STRING_PARAMETER_EXPECTED)
      return("STRING_PARAMETER_EXPECTED");
   if(code==ERR_INTEGER_PARAMETER_EXPECTED)
      return("INTEGER_PARAMETER_EXPECTED");
   if(code==ERR_DOUBLE_PARAMETER_EXPECTED)
      return("DOUBLE_PARAMETER_EXPECTED");
   if(code==ERR_ARRAY_AS_PARAMETER_EXPECTED)
      return("ARRAY_AS_PARAMETER_EXPECTED");
   if(code==ERR_HISTORY_WILL_UPDATED)
      return("HISTORY_WILL_UPDATED");
   if(code==ERR_TRADE_ERROR)
      return("TRADE_ERROR");
   if(code==ERR_RESOURCE_NOT_FOUND)
      return("RESOURCE_NOT_FOUND");
   if(code==ERR_RESOURCE_NOT_SUPPORTED)
      return("RESOURCE_NOT_SUPPORTED");
   if(code==ERR_RESOURCE_DUPLICATED)
      return("RESOURCE_DUPLICATED");
   if(code==ERR_INDICATOR_CANNOT_INIT)
      return("INDICATOR_CANNOT_INIT");
   if(code==ERR_END_OF_FILE)
      return("END_OF_FILE");
   if(code==ERR_SOME_FILE_ERROR)
      return("SOME_FILE_ERROR");
   if(code==ERR_WRONG_FILE_NAME)
      return("WRONG_FILE_NAME");
   if(code==ERR_TOO_MANY_OPENED_FILES)
      return("TOO_MANY_OPENED_FILES");
   if(code==ERR_CANNOT_OPEN_FILE)
      return("CANNOT_OPEN_FILE");
   if(code==ERR_INCOMPATIBLE_FILEACCESS)
      return("INCOMPATIBLE_FILEACCESS");
   if(code==ERR_NO_ORDER_SELECTED)
      return("NO_ORDER_SELECTED");
   if(code==ERR_UNKNOWN_SYMBOL)
      return("UNKNOWN_SYMBOL");
   if(code==ERR_INVALID_PRICE_PARAM)
      return("INVALID_PRICE_PARAM");
   if(code==ERR_INVALID_TICKET)
      return("INVALID_TICKET");
   if(code==ERR_TRADE_NOT_ALLOWED)
      return("TRADE_NOT_ALLOWED");
   if(code==ERR_LONGS_NOT_ALLOWED)
      return("LONGS_NOT_ALLOWED");
   if(code==ERR_SHORTS_NOT_ALLOWED)
      return("SHORTS_NOT_ALLOWED");
   if(code==ERR_TRADE_EXPERT_DISABLED_BY_SERVER)
      return("TRADE_EXPERT_DISABLED_BY_SERVER");
   if(code==ERR_OBJECT_ALREADY_EXISTS)
      return("OBJECT_ALREADY_EXISTS");
   if(code==ERR_UNKNOWN_OBJECT_PROPERTY)
      return("UNKNOWN_OBJECT_PROPERTY");
   if(code==ERR_OBJECT_DOES_NOT_EXIST)
      return("OBJECT_DOES_NOT_EXIST");
   if(code==ERR_UNKNOWN_OBJECT_TYPE)
      return("UNKNOWN_OBJECT_TYPE");
   if(code==ERR_NO_OBJECT_NAME)
      return("NO_OBJECT_NAME");
   if(code==ERR_OBJECT_COORDINATES_ERROR)
      return("OBJECT_COORDINATES_ERROR");
   if(code==ERR_NO_SPECIFIED_SUBWINDOW)
      return("NO_SPECIFIED_SUBWINDOW");
   if(code==ERR_SOME_OBJECT_ERROR)
      return("SOME_OBJECT_ERROR");
   if(code==ERR_CHART_PROP_INVALID)
      return("CHART_PROP_INVALID");
   if(code==ERR_CHART_NOT_FOUND)
      return("CHART_NOT_FOUND");
   if(code==ERR_CHARTWINDOW_NOT_FOUND)
      return("CHARTWINDOW_NOT_FOUND");
   if(code==ERR_CHARTINDICATOR_NOT_FOUND)
      return("CHARTINDICATOR_NOT_FOUND");
   if(code==ERR_SYMBOL_SELECT)
      return("SYMBOL_SELECT");
   if(code==ERR_NOTIFICATION_ERROR)
      return("NOTIFICATION_ERROR");
   if(code==ERR_NOTIFICATION_PARAMETER)
      return("NOTIFICATION_PARAMETER");
   if(code==ERR_NOTIFICATION_SETTINGS)
      return("NOTIFICATION_SETTINGS");
   if(code==ERR_NOTIFICATION_TOO_FREQUENT)
      return("NOTIFICATION_TOO_FREQUENT");
   if(code==ERR_FILE_TOO_MANY_OPENED)
      return("FILE_TOO_MANY_OPENED");
   if(code==ERR_FILE_WRONG_FILENAME)
      return("FILE_WRONG_FILENAME");
   if(code==ERR_FILE_TOO_LONG_FILENAME)
      return("FILE_TOO_LONG_FILENAME");
   if(code==ERR_FILE_CANNOT_OPEN)
      return("FILE_CANNOT_OPEN");
   if(code==ERR_FILE_BUFFER_ALLOCATION_ERROR)
      return("FILE_BUFFER_ALLOCATION_ERROR");
   if(code==ERR_FILE_CANNOT_DELETE)
      return("FILE_CANNOT_DELETE");
   if(code==ERR_FILE_INVALID_HANDLE)
      return("FILE_INVALID_HANDLE");
   if(code==ERR_FILE_WRONG_HANDLE)
      return("FILE_WRONG_HANDLE");
   if(code==ERR_FILE_NOT_TOWRITE)
      return("FILE_NOT_TOWRITE");
   if(code==ERR_FILE_NOT_TOREAD)
      return("FILE_NOT_TOREAD");
   if(code==ERR_FILE_NOT_BIN)
      return("FILE_NOT_BIN");
   if(code==ERR_FILE_NOT_TXT)
      return("FILE_NOT_TXT");
   if(code==ERR_FILE_NOT_TXTORCSV)
      return("FILE_NOT_TXTORCSV");
   if(code==ERR_FILE_NOT_CSV)
      return("FILE_NOT_CSV");
   if(code==ERR_FILE_READ_ERROR)
      return("FILE_READ_ERROR");
   if(code==ERR_FILE_WRITE_ERROR)
      return("FILE_WRITE_ERROR");
   if(code==ERR_FILE_BIN_STRINGSIZE)
      return("FILE_BIN_STRINGSIZE");
   if(code==ERR_FILE_INCOMPATIBLE)
      return("FILE_INCOMPATIBLE");
   if(code==ERR_FILE_IS_DIRECTORY)
      return("FILE_IS_DIRECTORY");
   if(code==ERR_FILE_NOT_EXIST)
      return("FILE_NOT_EXIST");
   if(code==ERR_FILE_CANNOT_REWRITE)
      return("FILE_CANNOT_REWRITE");
   if(code==ERR_FILE_WRONG_DIRECTORYNAME)
      return("FILE_WRONG_DIRECTORYNAME");
   if(code==ERR_FILE_DIRECTORY_NOT_EXIST)
      return("FILE_DIRECTORY_NOT_EXIST");
   if(code==ERR_FILE_NOT_DIRECTORY)
      return("FILE_NOT_DIRECTORY");
   if(code==ERR_FILE_CANNOT_DELETE_DIRECTORY)
      return("FILE_CANNOT_DELETE_DIRECTORY");
   if(code==ERR_FILE_CANNOT_CLEAN_DIRECTORY)
      return("FILE_CANNOT_CLEAN_DIRECTORY");
   if(code==ERR_FILE_ARRAYRESIZE_ERROR)
      return("FILE_ARRAYRESIZE_ERROR");
   if(code==ERR_FILE_STRINGRESIZE_ERROR)
      return("FILE_STRINGRESIZE_ERROR");
   if(code==ERR_FILE_STRUCT_WITH_OBJECTS)
      return("FILE_STRUCT_WITH_OBJECTS");
   if(code==ERR_WEBREQUEST_INVALID_ADDRESS)
      return("WEBREQUEST_INVALID_ADDRESS");
   if(code==ERR_WEBREQUEST_CONNECT_FAILED)
      return("WEBREQUEST_CONNECT_FAILED");
   if(code==ERR_WEBREQUEST_TIMEOUT)
      return("WEBREQUEST_TIMEOUT");
   if(code==ERR_WEBREQUEST_REQUEST_FAILED)
      return("WEBREQUEST_REQUEST_FAILED");
   return("NOT_DEFINED");
  }
//+------------------------------------------------------------------+
*/