﻿//+------------------------------------------------------------------+
//| SyncCrosshair.mq5                                                |
//| Multi-chart crosshair + scroll sync (MT5 Indicator)              |
//| Install: attach this indicator to each chart in the same group.  |
//+------------------------------------------------------------------+
#property strict
#property indicator_chart_window
#property indicator_plots 0

enum ENUM_SYNC_MODE
  {
   SyncMode_A = 0,   // 严格同步可视窗口（尽量一致）
   SyncMode_B = 1,   // 以锚点时间固定水平位置
   SyncMode_C = 2    // 只保证锚点时间可见
  };

input string        SyncGroup         = "A";      // 同组图表才同步
input bool          SyncHorizontalLine= true;     // 同步水平线
input bool          SyncScroll        = true;     // 同步滚动/缩放
input ENUM_SYNC_MODE SyncMode         = SyncMode_B; // 同步模式
input int           AnchorPosition    = 60;       // 0-100（从左侧的百分比）
input int           UpdateThrottleMs  = 30;       // 广播节流（毫秒）
input bool          SnapTimeToPeriod  = true;     // 时间对齐到当前周期K线开盘时间

input color         CrossColor        = clrDeepSkyBlue; // 十字线颜色
input int           CrossWidth        = 1;              // 十字线宽度
input ENUM_LINE_STYLE CrossStyle      = STYLE_DOT;      // 十字线样式

// 自定义指标必须存在；本指标不使用缓冲区
int OnCalculate(const int rates_total,
                const int prev_calculated,
                const datetime &time[],
                const double &open[],
                const double &high[],
                const double &low[],
                const double &close[],
                const long &tick_volume[],
                const long &volume[],
                const int &spread[])
  {
   return(rates_total);
  }

string  g_prefix;
string  g_obj_v;
string  g_obj_h;
string  g_key_seq, g_key_src, g_key_time, g_key_price, g_key_has_price, g_key_left, g_key_right;
string  g_key_instance;

long    g_last_seen_seq = 0;
long    g_last_send_ms  = 0;
long    g_suppress_until= 0;
int     g_throttle_ms   = 30;

bool    g_has_cursor    = false;
bool    g_has_price     = false;
datetime g_cursor_time  = 0;
double  g_cursor_price  = 0.0;

//+------------------------------------------------------------------+
int OnInit()
  {
   ChartSetInteger(0, CHART_EVENT_MOUSE_MOVE, true);

   g_prefix = "SYNC_XHAIR_" + SyncGroup + "_";
   g_obj_v  = "sync_xhair_v_" + IntegerToString((long)ChartID()) + "_" + SyncGroup;
   g_obj_h  = "sync_xhair_h_" + IntegerToString((long)ChartID()) + "_" + SyncGroup;

   g_key_seq       = g_prefix + "seq";
   g_key_src       = g_prefix + "src";
   g_key_time      = g_prefix + "time";
   g_key_price     = g_prefix + "price";
   g_key_has_price = g_prefix + "has_price";
   g_key_left      = g_prefix + "left_time";
   g_key_right     = g_prefix + "right_time";
   g_key_instance  = g_prefix + "instance_" + IntegerToString((long)ChartID());

   GlobalVariableSet(g_key_instance, (double)TimeLocal());

   g_throttle_ms = UpdateThrottleMs;
   if(g_throttle_ms < 10) g_throttle_ms = 10;
   EventSetMillisecondTimer(g_throttle_ms);

   return(INIT_SUCCEEDED);
  }

//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
   EventKillTimer();

   ObjectDelete(0, g_obj_v);
   ObjectDelete(0, g_obj_h);

   GlobalVariableDel(g_key_instance);
  }

//+------------------------------------------------------------------+
void OnTimer()
  {
   if(!GlobalVariableCheck(g_key_seq))
      return;

   long seq = (long)GlobalVariableGet(g_key_seq);
   if(seq == g_last_seen_seq)
      return;

   g_last_seen_seq = seq;

   long src = (long)GlobalVariableGet(g_key_src);
   if(src == (long)ChartID())
      return;

   datetime t = (datetime)GlobalVariableGet(g_key_time);
   if(t <= 0)
      return;

   double price   = GlobalVariableGet(g_key_price);
   bool has_price = (GlobalVariableGet(g_key_has_price) > 0.5);

   datetime left_time  = (datetime)GlobalVariableGet(g_key_left);
   datetime right_time = (datetime)GlobalVariableGet(g_key_right);

   g_suppress_until = GetTickCount() + (g_throttle_ms * 2);

   ApplyRemoteUpdate(t, price, has_price, left_time, right_time);
  }

//+------------------------------------------------------------------+
void OnChartEvent(const int id, const long &lparam, const double &dparam, const string &sparam)
  {
   if(GetTickCount() < g_suppress_until)
      return;

   if(id == CHARTEVENT_MOUSE_MOVE)
     {
      int x = (int)lparam;
      int y = (int)dparam;
      int subwin = 0;
      datetime t;
      double price;
     if(!ChartXYToTimePrice(0, x, y, subwin, t, price))
        return;

      t = SnapTimeToBarOpen(t);

      g_has_cursor   = true;
      g_has_price    = true;
      g_cursor_time  = t;
      g_cursor_price = price;

      UpdateLocalCrosshair(t, price, true);
      BroadcastIfNeeded(t, price, true);
     }
   else if(id == CHARTEVENT_CHART_CHANGE)
     {
      if(!SyncScroll)
         return;

      datetime t = 0;
      double price = g_cursor_price;
      bool has_price = g_has_price;

      if(g_has_cursor)
         t = g_cursor_time;
      else
         t = AnchorTimeFromVisibleRange();

      if(t <= 0)
         return;

      BroadcastIfNeeded(t, price, has_price);
     }
  }

//+------------------------------------------------------------------+
void BroadcastIfNeeded(datetime t, double price, bool has_price)
  {
   long now = GetTickCount();
   if(now - g_last_send_ms < g_throttle_ms)
      return;

   g_last_send_ms = now;

   datetime left_time = 0;
   datetime right_time = 0;
   GetVisibleRange(left_time, right_time);

   GlobalVariableSet(g_key_src, (double)ChartID());
   GlobalVariableSet(g_key_time, (double)t);
   GlobalVariableSet(g_key_price, price);
   GlobalVariableSet(g_key_has_price, has_price ? 1.0 : 0.0);
   GlobalVariableSet(g_key_left, (double)left_time);
   GlobalVariableSet(g_key_right, (double)right_time);
   GlobalVariableSet(g_key_seq, (double)GetMicrosecondCount());
  }

//+------------------------------------------------------------------+
void ApplyRemoteUpdate(datetime t, double price, bool has_price,
                       datetime left_time, datetime right_time)
  {
   // Always draw the vertical line at the exact source time
   UpdateLocalCrosshair(t, price, has_price);

   g_has_cursor   = true;
   g_has_price    = has_price;
   g_cursor_time  = t;
   if(has_price)
      g_cursor_price = price;

   if(SyncScroll)
      ApplyScrollSync(ResolveTimeToNearest(t), left_time, right_time);
  }

//+------------------------------------------------------------------+
void EnsureLineObjects()
  {
   if(ObjectFind(0, g_obj_v) < 0)
     {
      ObjectCreate(0, g_obj_v, OBJ_VLINE, 0, 0, 0);
      ObjectSetInteger(0, g_obj_v, OBJPROP_COLOR, CrossColor);
      ObjectSetInteger(0, g_obj_v, OBJPROP_WIDTH, CrossWidth);
      ObjectSetInteger(0, g_obj_v, OBJPROP_STYLE, CrossStyle);
      ObjectSetInteger(0, g_obj_v, OBJPROP_SELECTABLE, false);
      ObjectSetInteger(0, g_obj_v, OBJPROP_HIDDEN, true);
      ObjectSetInteger(0, g_obj_v, OBJPROP_BACK, false);
     }

   if(SyncHorizontalLine)
     {
      if(ObjectFind(0, g_obj_h) < 0)
        {
         ObjectCreate(0, g_obj_h, OBJ_HLINE, 0, 0, 0);
         ObjectSetInteger(0, g_obj_h, OBJPROP_COLOR, CrossColor);
         ObjectSetInteger(0, g_obj_h, OBJPROP_WIDTH, CrossWidth);
         ObjectSetInteger(0, g_obj_h, OBJPROP_STYLE, CrossStyle);
         ObjectSetInteger(0, g_obj_h, OBJPROP_SELECTABLE, false);
         ObjectSetInteger(0, g_obj_h, OBJPROP_HIDDEN, true);
         ObjectSetInteger(0, g_obj_h, OBJPROP_BACK, false);
        }
     }
  }

//+------------------------------------------------------------------+
void UpdateLocalCrosshair(datetime t, double price, bool has_price)
  {
   EnsureLineObjects();

   ObjectSetInteger(0, g_obj_v, OBJPROP_TIME, t);

   if(SyncHorizontalLine)
     {
      if(has_price)
        {
         ObjectSetDouble(0, g_obj_h, OBJPROP_PRICE, price);
         ObjectSetInteger(0, g_obj_h, OBJPROP_COLOR, CrossColor);
        }
      else
        {
         int bg = (int)ChartGetInteger(0, CHART_COLOR_BACKGROUND);
         ObjectSetInteger(0, g_obj_h, OBJPROP_COLOR, bg);
        }
     }

   ChartRedraw(0);
  }

//+------------------------------------------------------------------+
bool GetVisibleRange(datetime &left_time, datetime &right_time)
  {
   long left = (long)ChartGetInteger(0, CHART_FIRST_VISIBLE_BAR);
   long bars = (long)ChartGetInteger(0, CHART_VISIBLE_BARS);
   if(bars <= 0)
      return(false);

   long right = left - (bars - 1);
   if(right < 0) right = 0;

   left_time  = iTime(_Symbol, _Period, (int)left);
   right_time = iTime(_Symbol, _Period, (int)right);

   return(left_time > 0 && right_time > 0);
  }

//+------------------------------------------------------------------+
datetime AnchorTimeFromVisibleRange()
  {
   long left = (long)ChartGetInteger(0, CHART_FIRST_VISIBLE_BAR);
   long bars = (long)ChartGetInteger(0, CHART_VISIBLE_BARS);
   if(bars <= 0)
      return(0);

   int ap = AnchorPosition;
   if(ap < 0) ap = 0;
   if(ap > 100) ap = 100;
   double p = (double)ap / 100.0;
   long anchor = left - (long)MathRound((bars - 1) * p);
   if(anchor < 0) anchor = 0;

   datetime t = iTime(_Symbol, _Period, (int)anchor);
   return(t);
  }

//+------------------------------------------------------------------+
bool IsTimeVisible(datetime t)
  {
   datetime left_time, right_time;
   if(!GetVisibleRange(left_time, right_time))
      return(false);

   if(left_time <= right_time)
      return(t >= left_time && t <= right_time);
   return(t >= right_time && t <= left_time);
  }

//+------------------------------------------------------------------+
bool EnsureHistory(datetime t)
  {
   if(t <= 0)
      return(false);

   MqlRates rates[];
   int copied = CopyRates(_Symbol, _Period, t, 2, rates);
   if(copied > 0)
      return(true);

   int ps = PeriodSeconds(_Period);
   if(ps <= 0) ps = 60;
   copied = CopyRates(_Symbol, _Period, t - ps * 10, 2, rates);
   return(copied > 0);
  }

//+------------------------------------------------------------------+
datetime SnapTimeToBarOpen(datetime t)
  {
   if(!SnapTimeToPeriod || t <= 0)
      return(t);

   EnsureHistory(t);
   long shift = iBarShift(_Symbol, _Period, t, false);
   if(shift < 0)
      return(t);

   datetime bt = iTime(_Symbol, _Period, (int)shift);
   if(bt > 0)
      return(bt);

   return(t);
  }

//+------------------------------------------------------------------+
long GetBarShiftByTime(datetime t, bool exact)
  {
   if(t <= 0)
      return(-1);

   EnsureHistory(t);
   long shift = iBarShift(_Symbol, _Period, t, exact);
   if(shift < 0 && exact)
      shift = iBarShift(_Symbol, _Period, t, false);
   return(shift);
  }

//+------------------------------------------------------------------+
datetime ResolveTimeToNearest(datetime t)
  {
   long shift = GetBarShiftByTime(t, true);
   if(shift >= 0)
      return(iTime(_Symbol, _Period, (int)shift));

   // Fallback to the most recent available bar time
   datetime cur = iTime(_Symbol, _Period, 0);
   if(cur > 0)
     {
      Print("SyncCrosshair: no history for time ", TimeToString(t), ", fallback to latest bar");
      return(cur);
     }

   Print("SyncCrosshair: no history for time ", TimeToString(t), ", cannot resolve");
   return(t);
  }
//+------------------------------------------------------------------+
void ApplyScrollSync(datetime t, datetime left_time, datetime right_time)
  {
   if(!SyncScroll)
      return;

   if(SyncMode == SyncMode_A)
     {
      if(left_time > 0)
        {
         long left_shift = GetBarShiftByTime(left_time, true);
         if(left_shift >= 0)
            ChartSetInteger(0, CHART_FIRST_VISIBLE_BAR, left_shift);
        }
      else
         ApplyAnchorScroll(t);
     }
   else if(SyncMode == SyncMode_B)
     {
      ApplyAnchorScroll(t);
     }
   else if(SyncMode == SyncMode_C)
     {
      if(!IsTimeVisible(t))
         ApplyAnchorScroll(t);
     }
  }

//+------------------------------------------------------------------+
void ApplyAnchorScroll(datetime t)
  {
   long anchor_shift = GetBarShiftByTime(t, true);
   if(anchor_shift < 0)
      return;

   long bars = (long)ChartGetInteger(0, CHART_VISIBLE_BARS);
   if(bars <= 0)
      return;

   int ap = AnchorPosition;
   if(ap < 0) ap = 0;
   if(ap > 100) ap = 100;
   double p = (double)ap / 100.0;
   long left = anchor_shift + (long)MathRound((bars - 1) * p);
   if(left < 0) left = 0;
   ChartSetInteger(0, CHART_FIRST_VISIBLE_BAR, left);
  }

//+------------------------------------------------------------------+
