//+------------------------------------------------------------------+  
//| Multi-Currency Smart Dashboard EA                                |  
//| Fixed / improved version — TopWorldsHustler (original)           |  
//+------------------------------------------------------------------+  
#property strict  
#include <Trade/Trade.mqh>  
#include <Calendar.mqh>  
CTrade trade;  
  
//--- Inputs  
input string SymbolsToTrade = "EURUSD,GBPUSD,USDJPY";  
input ENUM_TIMEFRAMES TradeTF = PERIOD_M15;  
input ENUM_TIMEFRAMES TrendTF = PERIOD_H1;  
input int FastMA = 50;  
input int SlowMA = 200;  
input int RSIPeriod = 14;  
input int ATRPeriod = 14;  
input double ATRMultiplierLow = 1.2;  
input double ATRMultiplierHigh = 2.0;  
input double RiskPercent = 1.0;          // % of account equity risk per trade  
input double MaxSpread = 5.0;            // in pips (points)  
input double MaxLot = 1.0;  
input int NewsPauseMinutes = 30;  
input bool UseNewsFilter = true;  
input double BreakEvenProfitPips = 20.0;  
input double TP1Percent = 30.0;  
input double TP2Percent = 30.0;  
input double TP3Percent = 40.0;  
input double MaxDailyLossPercent = 5.0;  
  
//--- Panel (left as placeholders)  
int panel_x=20,panel_y=20,panel_width=500,panel_height=300;  
color bg_color=clrBlack, text_color=clrWhite;  
int font_size=11;  
string panel_name="DashboardPanel";  
string btnCloseAll="Btn_CloseAll";  
string btnToggleNews="Btn_ToggleNews";  
  
//--- Visuals placeholders  
string arrowBuy="Arrow_Buy";  
string arrowSell="Arrow_Sell";  
string barATR="Bar_ATR";  
string barRSI="Bar_RSI";  
color upColor=clrLime, downColor=clrRed;  
color atrColor=clrOrange, rsiColor=clrBlue;  
  
//--- Globals  
double StartingBalance;  
int ConsecutiveLosses=0;  
  
struct PairInfo  
{  
   string symbol;  
   double fastMA;  
   double slowMA;  
   double rsi;  
   double atr;  
   int trend;  
   double spread;    // in points  
   bool hasPosition;  
};  
PairInfo Pairs[];  
  
//---------------------------- Initialization  
int OnInit()  
{  
   StartingBalance = AccountInfoDouble(ACCOUNT_BALANCE);  
   InitPairs();  
   // Ensure symbols are selected/visible to platform  
   for(int i=0;i<ArraySize(Pairs);i++)  
   {  
      if(!SymbolSelect(Pairs[i].symbol,true))  
         PrintFormat("Warning: SymbolSelect failed for %s (might be unavailable)",Pairs[i].symbol);  
   }  
   DrawDashboardPanel();  
   DrawButtons();  
   return(INIT_SUCCEEDED);  
}  
  
//---------------------------- Initialize Pairs  
void InitPairs()  
{  
   string arr[];  
   int total = StringSplit(SymbolsToTrade, ',', arr);  
   ArrayResize(Pairs,total);  
   for(int i=0;i<total;i++)  
   {  
      arr[i] = StringTrim(arr[i]);  
      Pairs[i].symbol = arr[i];  
   }  
}  
  
//---------------------------- ATR & Lot Size  
double GetATR(string sym,ENUM_TIMEFRAMES tf,int period)  
{  
   // Use iATR current value (index 0)  
   return iATR(sym,tf,period,0);  
}  
double DynamicATRMultiplier(string sym)  
{  
   double atr = GetATR(sym,TrendTF,ATRPeriod);  
   double price = iClose(sym,TrendTF,0);  
   if(price<=0 || atr<=0) return ATRMultiplierLow;  
   return (atr/price>0.0015)?ATRMultiplierHigh:ATRMultiplierLow;  
}  
  
double LotSize(double stoploss_price,double takeprofit_price,string sym)  
{  
   // compute risk amount in account currency  
   double balance = AccountInfoDouble(ACCOUNT_BALANCE);  
   double risk = balance * RiskPercent / 100.0;  
   // Validate symbol trade parameters  
   double tick_size = SymbolInfoDouble(sym, SYMBOL_TRADE_TICK_SIZE);  
   double tick_value = SymbolInfoDouble(sym, SYMBOL_TRADE_TICK_VALUE);  
   if(tick_size <= 0 || tick_value <= 0)  
   {  
      PrintFormat("LotSize: invalid tick info for %s (tick_size=%f,tick_value=%f)", sym, tick_size, tick_value);  
      return 0.0;  
   }  
   double price_diff = MathAbs(takeprofit_price - stoploss_price);  
   if(price_diff <= 0)  
   {  
      PrintFormat("LotSize: zero price diff for %s", sym);  
      return 0.0;  
   }  
   // number of ticks in the SL distance  
   double ticks = price_diff / tick_size;  
   // value of that move (per 1.0 lot) = ticks * tick_value  
   double value_per_lot = ticks * tick_value;  
   if(value_per_lot <= 0)  
   {  
      PrintFormat("LotSize: zero value_per_lot %s", sym);  
      return 0.0;  
   }  
   double lot = risk / value_per_lot;  
   // clamp to broker limits  
   double maxLot = MathMin(MaxLot, SymbolInfoDouble(sym, SYMBOL_VOLUME_MAX));  
   double minLot = SymbolInfoDouble(sym, SYMBOL_VOLUME_MIN);  
   double step = SymbolInfoDouble(sym, SYMBOL_VOLUME_STEP);  
   if(step <= 0) step = 0.01;  
   // floor to step  
   int digits = (int)SymbolInfoInteger(sym, SYMBOL_VOLUME_DIGITS);  
   if(digits < 0) digits = 2;  
   lot = MathFloor(lot/step)*step;  
   lot = NormalizeDouble(lot, digits);  
   if(lot < minLot) lot = minLot;  
   if(lot > maxLot) lot = maxLot;  
   return lot;  
}  
  
//---------------------------- News Filter  
bool HighImpactNews()  
{  
   if(!UseNewsFilter) return false;  
   MqlCalendarEvent events[];  
   datetime from = TimeCurrent() - 60*60*24;  
   datetime to   = TimeCurrent() + 60*60*24;  
   int total = CalendarValueHistory(events, from, to);  
   if(total <= 0) return false;  
   for(int i=0;i<total;i++)  
   {  
      if(events[i].importance==CALENDAR_IMPORTANCE_HIGH)  
      {  
         if(MathAbs((long)(events[i].datetime - TimeCurrent())) <= NewsPauseMinutes*60)  
            return true;  
      }  
   }  
   return false;  
}  
  
//---------------------------- Higher TF Trend  
int HigherTimeframeTrend(string sym,ENUM_TIMEFRAMES tf)  
{  
   double fast = iMA(sym,tf,FastMA,0,MODE_SMA,PRICE_CLOSE,0);  
   double slow = iMA(sym,tf,SlowMA,0,MODE_SMA,PRICE_CLOSE,0);  
   if(fast==0 || slow==0) return 0;  
   if(fast>slow) return 1;  
   if(fast<slow) return -1;  
   return 0;  
}  
  
//---------------------------- Trailing & BreakEven  
void ApplyTrailingStop(string sym)  
{  
   int total = PositionsTotal();  
   for(int idx=total-1; idx>=0; idx--)  
   {  
      // get ticket and auto-select the position  
      ulong ticket = PositionGetTicket(idx);  
      if(ticket == 0) continue;  
      if(!PositionSelectByTicket(ticket)) continue;  
      string posSym = PositionGetString(POSITION_SYMBOL);  
      if(posSym != sym) continue;  
  
      int type = (int)PositionGetInteger(POSITION_TYPE);  
      double sl = PositionGetDouble(POSITION_SL);  
      double openPrice = PositionGetDouble(POSITION_PRICE_OPEN);  
      double cur = (type==POSITION_TYPE_BUY) ? SymbolInfoDouble(sym,SYMBOL_BID) : SymbolInfoDouble(sym,SYMBOL_ASK);  
      double atr = GetATR(sym,TradeTF,ATRPeriod) * DynamicATRMultiplier(sym);  
  
      // Break-even  
      double profitPips = (type==POSITION_TYPE_BUY) ? (cur - openPrice)/_Point : (openPrice - cur)/_Point;  
      if(profitPips >= BreakEvenProfitPips)  
      {  
         double beSL = openPrice;  
         // Use PositionModify by ticket; check result  
         bool modified = trade.PositionModify(ticket, beSL, PositionGetDouble(POSITION_TP));  
         if(!modified)  
            PrintFormat("PositionModify BE failed for ticket %d sym %s error %d", ticket, sym, GetLastError());  
      }  
  
      // Trailing using ATR (only modify if newSL is more protective)  
      if(atr > 0)  
      {  
         if(type==POSITION_TYPE_BUY)  
         {  
            double newSL = NormalizeDouble(cur - atr, _Digits);  
            if(newSL > sl && (cur - openPrice) > atr)  
            {  
               if(!trade.PositionModify(ticket, newSL, PositionGetDouble(POSITION_TP)))  
                  PrintFormat("PositionModify trailing failed (BUY) ticket %d err %d", ticket, GetLastError());  
            }  
         }  
         else // SELL  
         {  
            double newSL = NormalizeDouble(cur + atr, _Digits);  
            // sl==0 denotes no SL (or invalid) so check  
            if((sl==0.0 || newSL < sl) && (openPrice - cur) > atr)  
            {  
               if(!trade.PositionModify(ticket, newSL, PositionGetDouble(POSITION_TP)))  
                  PrintFormat("PositionModify trailing failed (SELL) ticket %d err %d", ticket, GetLastError());  
            }  
         }  
      }  
   }  
}  
  
//---------------------------- Partial TP  
void CheckPartialTP(string sym)  
{  
   int total = PositionsTotal();  
   for(int idx=total-1; idx>=0; idx--)  
   {  
      ulong ticket = PositionGetTicket(idx);  
      if(ticket == 0) continue;  
      if(!PositionSelectByTicket(ticket)) continue;  
      string posSym = PositionGetString(POSITION_SYMBOL);  
      if(posSym != sym) continue;  
  
      int type = (int)PositionGetInteger(POSITION_TYPE);  
      double openPrice = PositionGetDouble(POSITION_PRICE_OPEN);  
      double lots = PositionGetDouble(POSITION_VOLUME);  
      double cur = (type==POSITION_TYPE_BUY) ? SymbolInfoDouble(sym,SYMBOL_BID) : SymbolInfoDouble(sym,SYMBOL_ASK);  
  
      double atr = GetATR(sym,TradeTF,ATRPeriod) * DynamicATRMultiplier(sym);  
      if(atr<=0) continue;  
      double tp1 = (type==POSITION_TYPE_BUY) ? openPrice + atr : openPrice - atr;  
      double tp2 = (type==POSITION_TYPE_BUY) ? openPrice + atr*2 : openPrice - atr*2;  
      double tp3 = (type==POSITION_TYPE_BUY) ? openPrice + atr*3 : openPrice - atr*3;  
  
      // Close partials - note: PositionClosePartial has some broker/platform caveats  
      double v1 = lots * TP1Percent / 100.0;  
      double v2 = lots * TP2Percent / 100.0;  
      double v3 = lots * TP3Percent / 100.0;  
  
      if((type==POSITION_TYPE_BUY && cur >= tp1) || (type==POSITION_TYPE_SELL && cur <= tp1))  
      {  
         if(v1 > 0)  
            if(!trade.PositionClosePartial(sym, v1))  
               PrintFormat("Partial close TP1 failed for %s ticket %d err %d", sym, ticket, GetLastError());  
      }  
      if((type==POSITION_TYPE_BUY && cur >= tp2) || (type==POSITION_TYPE_SELL && cur <= tp2))  
      {  
         if(v2 > 0)  
            if(!trade.PositionClosePartial(sym, v2))  
               PrintFormat("Partial close TP2 failed for %s ticket %d err %d", sym, ticket, GetLastError());  
      }  
      if((type==POSITION_TYPE_BUY && cur >= tp3) || (type==POSITION_TYPE_SELL && cur <= tp3))  
      {  
         if(v3 > 0)  
            if(!trade.PositionClosePartial(sym, v3))  
               PrintFormat("Partial close TP3 failed for %s ticket %d err %d", sym, ticket, GetLastError());  
      }  
   }  
}  
  
//---------------------------- Multi-Pair Trading Logic  
void OnTick()  
{  
   if(UseNewsFilter && HighImpactNews())   
   {  
      // if news, skip opening new trades (but manage existing ones)  
      // don't return outright if you want trailing/partials to continue; just skip openings  
      // return;  // original code returned; I prefer to only skip openings  
   }  
  
   for(int i=0;i<ArraySize(Pairs);i++)  
   {  
      string sym = Pairs[i].symbol;  
      // guard: ensure symbol exists  
      if(!SymbolInfoInteger(sym,SYMBOL_EXISTS))  
      {  
         // try to select  
         if(!SymbolSelect(sym, true))  
         {  
            PrintFormat("Symbol %s not available, skipping", sym);  
            continue;  
         }  
      }  
  
      Pairs[i].fastMA = iMA(sym,TradeTF,FastMA,0,MODE_SMA,PRICE_CLOSE,0);  
      Pairs[i].slowMA = iMA(sym,TradeTF,SlowMA,0,MODE_SMA,PRICE_CLOSE,0);  
      Pairs[i].rsi = iRSI(sym,TradeTF,RSIPeriod,PRICE_CLOSE,0);  
      Pairs[i].atr = GetATR(sym,TradeTF,ATRPeriod)*DynamicATRMultiplier(sym);  
      Pairs[i].trend = HigherTimeframeTrend(sym,TrendTF);  
      Pairs[i].spread = (SymbolInfoDouble(sym,SYMBOL_ASK)-SymbolInfoDouble(sym,SYMBOL_BID))/_Point;  
      Pairs[i].hasPosition = (PositionsTotalBySymbol(sym)>0);  
  
      // opening logic: only open if no position and spread OK  
      if(!Pairs[i].hasPosition && Pairs[i].spread <= MaxSpread)  
      {  
         // BUY  
         if(Pairs[i].fastMA > Pairs[i].slowMA && Pairs[i].rsi > 50 && Pairs[i].trend >= 0)  
         {  
            double ask = SymbolInfoDouble(sym,SYMBOL_ASK);  
            double sl = ask - Pairs[i].atr;  
            double tp = ask + Pairs[i].atr*3.0;  
            // validate stop distance vs broker stops level  
            int stops_level = (int)SymbolInfoInteger(sym,SYMBOL_TRADE_STOPS_LEVEL);  
            double minDistance = (double)stops_level * _Point;  
            if(MathAbs(ask - sl) < minDistance)  
            {  
               PrintFormat("Buy: SL too close for %s, skip (minStops=%d pts)", sym, stops_level);  
            }  
            else  
            {  
               double lot = LotSize(sl, tp, sym);  
               if(lot > 0)  
               {  
                  bool ok = trade.Buy(lot, sym, ask, sl, tp, "EA_Buy");  
                  if(!ok) PrintFormat("Buy failed %s err=%d", sym, GetLastError());  
               }  
            }  
         }  
  
         // SELL  
         if(Pairs[i].fastMA < Pairs[i].slowMA && Pairs[i].rsi < 50 && Pairs[i].trend <= 0)  
         {  
            double bid = SymbolInfoDouble(sym,SYMBOL_BID);  
            double sl = bid + Pairs[i].atr;  
            double tp = bid - Pairs[i].atr*3.0;  
            int stops_level = (int)SymbolInfoInteger(sym,SYMBOL_TRADE_STOPS_LEVEL);  
            double minDistance = (double)stops_level * _Point;  
            if(MathAbs(bid - sl) < minDistance)  
            {  
               PrintFormat("Sell: SL too close for %s, skip (minStops=%d pts)", sym, stops_level);  
            }  
            else  
            {  
               double lot = LotSize(sl, tp, sym);  
               if(lot > 0)  
               {  
                  bool ok = trade.Sell(lot, sym, bid, sl, tp, "EA_Sell");  
                  if(!ok) PrintFormat("Sell failed %s err=%d", sym, GetLastError());  
               }  
            }  
         }  
      } // end open check  
  
      // Manage existing positions (trailing, partials)  
      ApplyTrailingStop(sym);  
      CheckPartialTP(sym);  
   }  
  
   // Dashboard visuals (lightweight redraw)  
   DrawDashboardPanel();  
   DrawButtons();  
   DrawArrowsBars();  
}  
  
//---------------------------- Dashboard & Visuals (Implement as needed)  
void DrawDashboardPanel(){ /* TODO: draw all pairs info; keep lightweight in OnTick */ }  
void DrawButtons(){ /* TODO: create close-all / toggle news UI objects */ }  
void DrawArrowsBars(){ /* TODO: arrows & ATR/RSI bars per pair */ }  
  
//---------------------------- Chart Event Handling  
void OnChartEvent(const int id,const long &lparam,const double &dparam,const string &sparam)  
{  
   if(id==CHARTEVENT_OBJECT_CLICK)  
   {  
      if(sparam==btnCloseAll)  
      {  
         // Collect tickets then close to avoid indexing issues while closing  
         int total = PositionsTotal();  
         ulong tickets[];  
         ArrayResize(tickets, total);  
         int cnt=0;  
         for(int i=0;i<total;i++)  
         {  
            ulong t = PositionGetTicket(i);  
            if(t>0) tickets[cnt++] = t;  
         }  
         for(int j=0;j<cnt;j++)  
         {  
            if(!trade.PositionClose(tickets[j])) PrintFormat("Close failed for ticket %d err %d", tickets[j], GetLastError());  
         }  
         Print("All trades close attempted.");  
      }  
      if(sparam==btnToggleNews)  
      {  
         UseNewsFilter = !UseNewsFilter;  
         // update object text safely if exists  
         if(ObjectFind(0, btnToggleNews) != -1)  
            ObjectSetString(0, btnToggleNews, OBJPROP_TEXT, UseNewsFilter ? "News: ON" : "News: OFF");  
         PrintFormat("News filter toggled: %s", UseNewsFilter ? "ON" : "OFF");  
      }  
   }  
}  
  
//---------------------------- Helper  
int PositionsTotalBySymbol(string sym)  
{  
   int count=0;  
   int total = PositionsTotal();  
   for(int i=0;i<total;i++)  
   {  
      ulong ticket = PositionGetTicket(i); // this auto-selects  
      if(ticket==0) continue;  
      string psym = PositionGetString(POSITION_SYMBOL);  
      if(psym == sym) count++;  
   }  
   return count;  
}  
