//+------------------------------------------------------------------+
//|                                                  Momentum_V1.mq5 |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#property version   "2.00"
//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+


#include <Trade\Trade.mqh>

CTrade trade;
// Inputs
input ENUM_TIMEFRAMES Timeframe = PERIOD_M15;   // Trading Timeframe
input long     MagicNumber       = 151515;      // Magic Number
input string   TradingStartTime  = "05:00";     // Trading Start Time
input string   TradingEndTime    = "20:00";     // Trading End Time
input double   MainTradeLotSize  = 0.02;        // MT Lot Size
input int      MainTradeTP       = 50;          // MT Take Profit
input int      CounterTradeTP    = 50;          // CT Take Profit
input bool     special_rule      = false;       // IF True, Then Use Margin %
input double   loss_threshold    = 50;          // IF $Loss > $X, Then use %
input double   MinCTProfit       = 5.0;         // Minimum CT profit in dollars
input double   MinCTProfitPct    = 10.0;        // Minimum CT profit in percentage
input int      SpreadFilter      = 30;          // Maximum spread filter in points
input int      StopLossLevel     = 0;           // Stop loss level in points
input double   MaxBuyLots        = 10.00;       // Maximum buy active trade lots
input double   MaxSellLots       = 10.00;       // Maximum sell active trade lots


int bar=0;
ulong ct_buy=0,ct_sell=0;
ulong mt_buy=0,mt_sell=0;

int OnInit()
  {
//---
   trade.SetExpertMagicNumber(MagicNumber);
//---
   return(INIT_SUCCEEDED);
  }
//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
//---
   
  }
//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
  {
//---
   //close trades
   if(ct_buy!=0 && !PositionSelectByTicket(ct_buy))
   {
      double profit = get_profit(ct_buy);
      if(PositionSelectByTicket(mt_sell))
      {
         
         double to_close = PositionGetDouble(POSITION_VOLUME)*(profit/PositionGetDouble(POSITION_PROFIT));
         to_close=MathAbs(to_close);
         to_close=NormalizeDouble(to_close,2);Print("TO CLOSE",to_close);
         bool closed_all = to_close>=PositionGetDouble(POSITION_VOLUME);
         if(!trade.PositionClosePartial(mt_sell,to_close))
         {
            Print("CLOSE PARTIAL FOR MT SELL FAILED ",trade.ResultRetcode());
         }
         else
         {
            mt_sell= closed_all ? 0 : mt_sell;
         }
         
      }
      if(PositionSelectByTicket(ct_sell))
      {
         
         double to_close = PositionGetDouble(POSITION_VOLUME)*(profit/PositionGetDouble(POSITION_PROFIT));
         to_close=MathAbs(to_close);
         to_close=NormalizeDouble(to_close,2);Print("TO CLOSE",to_close);
         bool closed_all = to_close>=PositionGetDouble(POSITION_VOLUME);
         if(!trade.PositionClosePartial(ct_sell,to_close))
         {
            Print("CLOSE PARTIAL FOR CT SELL FAILED ",trade.ResultRetcode());
         }
         else
         {
            ct_sell= closed_all ? 0 : ct_sell;
         }
      }
      ct_buy=0;
   }
   if(ct_sell!=0 && !PositionSelectByTicket(ct_sell))
   {
      double profit = get_profit(ct_sell);
      if(PositionSelectByTicket(mt_buy))
      {
         
         double to_close = PositionGetDouble(POSITION_VOLUME)*(profit/PositionGetDouble(POSITION_PROFIT));
         to_close=MathAbs(to_close);
         to_close=NormalizeDouble(to_close,2);Print("TO CLOSE",to_close);
         bool closed_all = to_close>=PositionGetDouble(POSITION_VOLUME);
         if(!trade.PositionClosePartial(mt_buy,to_close))
         {
            Print("CLOSE PARTIAL FOR MT BUY FAILED ",trade.ResultRetcode());
         }
         else
         {
            mt_buy= closed_all ? 0 : mt_buy;
         }
      }
      if(PositionSelectByTicket(ct_buy))
      {
         
         double to_close = PositionGetDouble(POSITION_VOLUME)*(profit/PositionGetDouble(POSITION_PROFIT));
         to_close=MathAbs(to_close);
         to_close=NormalizeDouble(to_close,2);Print("TO CLOSE",to_close);
         bool closed_all = to_close>=PositionGetDouble(POSITION_VOLUME);
         if(!trade.PositionClosePartial(ct_buy,to_close))
         {
            Print("CLOSE PARTIAL FOR CT BUY FAILED ",trade.ResultRetcode());
         }
         else
         {
            ct_buy= closed_all ? 0 : ct_buy;
         }
      }
      ct_sell=0;
   }

      if(bar!=iBars(_Symbol,Timeframe))
      {
         if(bar==0)
         {
            bar=iBars(_Symbol,Timeframe);
            return;
         }
         bar=iBars(_Symbol,Timeframe);
         
         //MT
         if( true )
         {
            //buy
            if(!IsBuyPositionOpened(MagicNumber) && ( !PositionSelectByTicket(mt_sell) || get_profit_live(mt_sell)>0 ) && ( !PositionSelectByTicket(ct_sell) || get_profit_live(ct_sell)>0 ) && buy_signal() && filter())
            {
               if(!trade.Buy(MathMin(MainTradeLotSize,MaxBuyLots),NULL,0,0,0,ConvertTimeframeToString(Timeframe)+" Momentum MT"))
               {
                  Print("BUY TRADE FAILED ",trade.ResultRetcode());
               }
               else
               {
                  ulong tick = trade.ResultOrder();
                  add_tp(tick,1,CalculateTakeProfitPrice(tick,1,MainTradeTP));
                  add_sl(tick,1,StopLossLevel);
                  mt_buy=tick;
               }
               
            }
            
            //sell
            if(!IsSellPositionOpened(MagicNumber) && ( !PositionSelectByTicket(mt_buy) || get_profit_live(mt_buy)>0 ) && ( !PositionSelectByTicket(ct_buy) || get_profit_live(ct_buy)>0 ) && sell_signal() && filter())
            {
               if(!trade.Sell(MathMin(MainTradeLotSize,MaxSellLots),NULL,0,0,0,ConvertTimeframeToString(Timeframe)+" Momentum MT"))
               {
                  Print("SELL TRADE FAILED ",trade.ResultRetcode());
               }
               else
               {
                  ulong tick = trade.ResultOrder();
                  add_tp(tick,-1,CalculateTakeProfitPrice(tick,-1,MainTradeTP));
                  add_sl(tick,-1,StopLossLevel);
                  mt_sell=tick;
               }
            }
         }
         
         //CT
         if( true )
         {
            //buy
            if(!IsBuyPositionOpened(MagicNumber) && IsSellPositionOpened(MagicNumber) && ( get_profit_live(mt_sell)<0 || get_profit_live(ct_sell)<0 ) && buy_signal() && filter())
            {
               if(!trade.Buy(MathMin(ctLotSize(1,SymbolInfoDouble(_Symbol,SYMBOL_ASK)+_Point*CounterTradeTP),MaxBuyLots),NULL,0,0,0,ConvertTimeframeToString(Timeframe)+" Momentum CT"))
               {
                  Print("BUY CT TRADE FAILED ",trade.ResultRetcode());
               }
               else
               {
                  ulong tick = trade.ResultOrder();
                  add_tp(tick,1,CalculateTakeProfitPrice(tick,1,CounterTradeTP));
                  add_sl(tick,1,StopLossLevel);
                  ct_buy=tick;
               }
            }
            
            //sell
            if(!IsSellPositionOpened(MagicNumber) && IsBuyPositionOpened(MagicNumber) && ( get_profit_live(mt_buy)<0 || get_profit_live(ct_buy)<0 ) && sell_signal() && filter())
            {
               if(!trade.Sell(MathMin(ctLotSize(-1,SymbolInfoDouble(_Symbol,SYMBOL_BID)-_Point*CounterTradeTP),MaxSellLots),NULL,0,0,0,ConvertTimeframeToString(Timeframe)+" Momentum CT"))
               {
                  Print("SELL CT TRADE FAILED ",trade.ResultRetcode());
               }
               else
               {
                  ulong tick = trade.ResultOrder();
                  add_tp(tick,-1,CalculateTakeProfitPrice(tick,-1,CounterTradeTP));
                  add_sl(tick,-1,StopLossLevel);
                  ct_sell=tick;
               }
            }
            
         }
         
      }

   
  }
//+------------------------------------------------------------------+

bool buy_signal()
{
   if(iClose(_Symbol,Timeframe,1)>iOpen(_Symbol,Timeframe,1))
   {
      return true;
   }
   return false;
}

bool sell_signal()
{
   if(iClose(_Symbol,Timeframe,1)<iOpen(_Symbol,Timeframe,1))
   {
      return true;
   }
   return false;
}



bool IsTimeInRange(string startTime, string endTime) {
   
   StringReplace(startTime," ","");
   StringReplace(endTime," ","");

    // Convert input time strings to datetime
    datetime start = StringToTime(TimeToString(TimeCurrent(), TIME_DATE) + " " + startTime);
    datetime end = StringToTime(TimeToString(TimeCurrent(), TIME_DATE) + " " + endTime);
    datetime currentTime = TimeCurrent();

    // Check if the current time is within the specified range
    if (start < end) {
        // Normal case: start and end times are on the same day
        return (currentTime >= start && currentTime <= end);
    } else {
        // Overnight case: end time is on the next day
        return (currentTime >= start || currentTime <= end);
    }
}


//+------------------------------------------------------------------+
//| Function to check if the current spread is within the limit      |
//+------------------------------------------------------------------+
bool IsSpreadWithinLimit(int maxSpread)
{
    // Get the current spread in points
    int spread = (int)SymbolInfoInteger(_Symbol, SYMBOL_SPREAD);

    // Check if the spread is less than or equal to the maximum spread
    if(spread <= maxSpread)
    {
        return true;  // Spread is within the limit
    }
        return false; // Spread exceeds the limit
}




//+------------------------------------------------------------------+
//| Function to calculate the take profit price                      |
//| type: 1 for buy, -1 for sell                                     |
//| takeProfitPoints: take profit in points                          |
//+------------------------------------------------------------------+
double CalculateTakeProfitPrice(ulong tick, int type, int takeProfitPoints)
{
   if(takeProfitPoints==0)
   {
      return 0;
   }
    // Get the current market price
    if(!PositionSelectByTicket(tick))
    {
      Print("FAILED TO SELECT POSITION TO MODIFY TP");
      return 0;
    }
    
    double currentPrice = PositionGetDouble(POSITION_PRICE_OPEN);
    
    // Get the point size for the current symbol
    double pointSize = SymbolInfoDouble(_Symbol, SYMBOL_POINT);
    
    // Calculate the take profit price
    double takeProfitPrice;
    if (type == 1) // Buy
    {
        takeProfitPrice = currentPrice + (takeProfitPoints * pointSize);
    }
    else if (type == -1) // Sell
    {
        takeProfitPrice = currentPrice - (takeProfitPoints * pointSize);
    }
    else
    {
        // Invalid trade type
        Print("Invalid trade type. Use 1 for buy and -1 for sell.");
        return 0.0;
    }
    
    return takeProfitPrice;
}



//+------------------------------------------------------------------+
//| Function to check if there's a buy position with a specific     |
//| magic number currently opened.                                  |
//| Input: magicNumber - the magic number to check for               |
//| Returns: true if a buy position with the given magic number is   |
//|          found, false otherwise                                  |
//+------------------------------------------------------------------+
bool IsBuyPositionOpened(long magicNumber)
{
    // Get the total number of open positions
    int totalPositions = PositionsTotal();

    // Loop through all open positions
    for (int i = 0; i < totalPositions; i++)
    {
        // Get the position ticket
        ulong positionTicket = PositionGetTicket(i);

        // Select the position by ticket
        if (PositionSelectByTicket(positionTicket))
        {
            // Check if the position is a buy position and matches the magic number
            if (PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY &&
                PositionGetInteger(POSITION_MAGIC) == magicNumber)
            {
                // A buy position with the specified magic number is found
                return true;
            }
        }
    }

    // No buy position with the specified magic number is found
    return false;
}

bool IsSellPositionOpened(long magicNumber)
{
    // Get the total number of open positions
    int totalPositions = PositionsTotal();

    // Loop through all open positions
    for (int i = 0; i < totalPositions; i++)
    {
        // Get the position ticket
        ulong positionTicket = PositionGetTicket(i);

        // Select the position by ticket
        if (PositionSelectByTicket(positionTicket))
        {
            // Check if the position is a buy position and matches the magic number
            if (PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_SELL &&
                PositionGetInteger(POSITION_MAGIC) == magicNumber)
            {
                // A buy position with the specified magic number is found
                return true;
            }
        }
    }

    // No buy position with the specified magic number is found
    return false;
}

bool filter()
{
   if(IsTimeInRange(TradingStartTime,TradingEndTime) && IsSpreadWithinLimit(SpreadFilter)  )
   {
      return true;
   }
   return false;
}


double ctLotSize(int typ,double open)
{
   if(typ==1)
   {
      double open_price = 0;
      double lots = 0;
      double dollar_loss=0;
      for(int i=PositionsTotal()-1;i>=0;i--)
      {
         if(PositionGetTicket(i) && PositionGetInteger(POSITION_MAGIC)==MagicNumber && PositionGetInteger(POSITION_TYPE)==POSITION_TYPE_SELL)
         {
            open_price=PositionGetDouble(POSITION_PRICE_OPEN);
            lots = PositionGetDouble(POSITION_VOLUME);
            dollar_loss=PositionGetDouble(POSITION_PROFIT);
            break;
         }
      }
      
      if(open_price==0)
      {
         Print("No opened sell trade");
         return 0;
      }
      
      double loss = MathAbs(open - open_price)*100*lots;
      double total_loss = special_rule && dollar_loss<0 && MathAbs(dollar_loss)>=loss_threshold ? (1+MinCTProfitPct/100)*loss :  MinCTProfit >0 ? loss + MinCTProfit : (1+MinCTProfitPct/100)*loss ;
      total_loss = MathRound(total_loss);
      Print("BUY CT ",total_loss," LOSS ",loss," DIF ",MathAbs(open - open_price)," lot ",lots," LOT ",CounterTradeTP <= 0 ? 0 : total_loss/CounterTradeTP);
      return NormalizeDouble(CounterTradeTP <= 0 ? 0 : total_loss/CounterTradeTP,2);
      
   }
   
   if(typ==-1)
   {
      double open_price = 0;
      double lots = 0;
      double dollar_loss=0;
      for(int i=PositionsTotal()-1;i>=0;i--)
      {
         if(PositionGetTicket(i) && PositionGetInteger(POSITION_MAGIC)==MagicNumber && PositionGetInteger(POSITION_TYPE)==POSITION_TYPE_BUY)
         {
            open_price=PositionGetDouble(POSITION_PRICE_OPEN);
            lots = PositionGetDouble(POSITION_VOLUME);
            dollar_loss=PositionGetDouble(POSITION_PROFIT);
            break;
         }
      }
      
      if(open_price==0)
      {
         Print("No opened buy trade");
         return 0;
      }
      
      double loss = MathAbs(open - open_price)*100*lots;
      double total_loss = special_rule && dollar_loss<0 && MathAbs(dollar_loss)>=loss_threshold ? (1+MinCTProfitPct/100)*loss : MinCTProfit >0 ? loss + MinCTProfit : (1+MinCTProfitPct/100)*loss ;
      total_loss = MathRound(total_loss);
      Print("SELL CT ",total_loss," LOSS ",loss," DIF ",MathAbs(open - open_price)," lot ",lots," LOT ",CounterTradeTP <= 0 ? 0 : total_loss/CounterTradeTP);
      return NormalizeDouble(CounterTradeTP <= 0 ? 0 : total_loss/CounterTradeTP,2);
      
   }
   return 0;
}

void add_sl(ulong ticket,int typ,int sl)
{
   if(sl==0)
   {
      return;
   }

   if(!PositionSelectByTicket(ticket))
   {
      Print(__FUNCTION__ ," FAILED . COULDN'T SELECT POSITION ",ticket);
      return;
   }
   
   if(PositionGetInteger(POSITION_TYPE)==POSITION_TYPE_BUY)
   {
      if(!trade.PositionModify(ticket,sl!=0 ? NormalizeDouble(PositionGetDouble(POSITION_PRICE_OPEN)-sl*_Point ,_Digits) : 0 ,PositionGetDouble(POSITION_TP)))
      {
         double indent = (double) MathMax(SymbolInfoInteger(_Symbol,SYMBOL_TRADE_FREEZE_LEVEL),SymbolInfoInteger(_Symbol,SYMBOL_TRADE_STOPS_LEVEL));
         Print(__FUNCTION__," FAILED TO SET SL FOR BUY ERROR ",trade.ResultRetcode()," SL should be below or equal to ",SymbolInfoDouble(_Symbol,SYMBOL_BID)-indent*_Point);
      }
      else
      {
         Print(__FUNCTION__," "+IntegerToString(ticket)+" SL PLACED");
      }
   }
   if(PositionGetInteger(POSITION_TYPE)==POSITION_TYPE_SELL)
   {
      if(!trade.PositionModify(ticket,sl!=0 ? NormalizeDouble(PositionGetDouble(POSITION_PRICE_OPEN)+sl*_Point,_Digits) : 0 ,PositionGetDouble(POSITION_TP)))
      {
         double indent = (double) MathMax(SymbolInfoInteger(_Symbol,SYMBOL_TRADE_FREEZE_LEVEL),SymbolInfoInteger(_Symbol,SYMBOL_TRADE_STOPS_LEVEL));
         Print(__FUNCTION__," FAILED TO SET SL FOR SELL ERROR ",trade.ResultRetcode()," SL should be above or equal to ",SymbolInfoDouble(_Symbol,SYMBOL_ASK)+indent*_Point);
      }
      else
      {
         Print(__FUNCTION__," "+IntegerToString(ticket)+" SL PLACED");
      }
   }
   
}

void add_tp(ulong ticket, int typ,double tp)
{
   if(!PositionSelectByTicket(ticket))
   {
      Print(__FUNCTION__ ," FAILED . COULDN'T SELECT POSITION ",ticket);
      return;
   }
   
   if(PositionGetInteger(POSITION_TYPE)==POSITION_TYPE_BUY)
   {
      if(!trade.PositionModify(ticket, PositionGetDouble(POSITION_SL) ,tp!=0 ? NormalizeDouble(tp,_Digits) : 0 ))
      {
         double indent = (double) MathMax(SymbolInfoInteger(_Symbol,SYMBOL_TRADE_FREEZE_LEVEL),SymbolInfoInteger(_Symbol,SYMBOL_TRADE_STOPS_LEVEL));
         Print(__FUNCTION__," FAILED TO SET TP FOR BUY ERROR ",trade.ResultRetcode()," TP should be above or equal to ",SymbolInfoDouble(_Symbol,SYMBOL_ASK)+indent*_Point);
      }
      else
      {
         Print(__FUNCTION__," "+IntegerToString(ticket)+" TP PLACED");
      }
   }
   if(PositionGetInteger(POSITION_TYPE)==POSITION_TYPE_SELL)
   {
      if(!trade.PositionModify(ticket, PositionGetDouble(POSITION_SL) ,tp!=0 ? NormalizeDouble(tp,_Digits) : 0 ))
      {
         double indent = (double) MathMax(SymbolInfoInteger(_Symbol,SYMBOL_TRADE_FREEZE_LEVEL),SymbolInfoInteger(_Symbol,SYMBOL_TRADE_STOPS_LEVEL));
         Print(__FUNCTION__," FAILED TO SET TP FOR SELL ERROR ",trade.ResultRetcode()," TP should be below or equal to ",SymbolInfoDouble(_Symbol,SYMBOL_BID)-indent*_Point);
      }
      else
      {
         Print(__FUNCTION__," "+IntegerToString(ticket)+" TP PLACED");
      }
   }
   
}

double get_profit(long position_ticket)
  {
//--- request the history of deals and orders for the specified position
   if(!HistorySelectByPosition(position_ticket))
     {
      Print(__FUNCTION__+" > Error: failed to select position ticket #",position_ticket,". Error Code: ", GetLastError());
      return(DBL_MIN);
     }
//---
   CDealInfo deal;
   string    pos_symbol=NULL;
   long      pos_id=-1;
   long      pos_type=-1;
   long      pos_magic=-1;
   double    pos_open_price=0;
   double    pos_close_price=0;
   double    pos_sl = 0;
   double    pos_tp = 0;
   double    pos_commission = 0;
   double    pos_swap=0;
   double    pos_profit=0;
   double    pos_open_volume= 0;
   double    pos_close_volume=0;
   datetime  pos_open_time=0;
   datetime  pos_close_time=0;
   double    pos_sum_cost=0;
   long      pos_open_reason=-1;
   long      pos_close_reason=-1;
   string    pos_open_comment = NULL;
   string    pos_close_comment = NULL;
   string    pos_deal_in = NULL;
   string    pos_deal_out=NULL;

//--- now process the list of received deals for the specified position
   int deals=HistoryDealsTotal();
   for(int i=0; i<deals && !IsStopped(); i++)
     {
      if(!deal.SelectByIndex(i))
        {
         Print(__FUNCTION__+" > Error: failed to select deal at index #", i);
         return(DBL_MIN);
        }
      //---
      pos_id              = deal.PositionId();
      pos_symbol          = deal.Symbol();
      pos_commission     += deal.Commission();
      pos_swap           += deal.Swap();
      pos_profit         += deal.Profit();
      //--- Entry deal for position
      if(deal.Entry()==DEAL_ENTRY_IN)
        {
         pos_magic           = deal.Magic();
         pos_type            = deal.DealType();
         pos_open_time       = deal.Time();
         pos_open_price      = deal.Price();
         pos_open_volume     = deal.Volume();
         //---
         pos_open_comment    = deal.Comment();
         pos_deal_in         = IntegerToString(deal.Ticket());
         pos_open_reason     = HistoryDealGetInteger(deal.Ticket(), DEAL_REASON);
        }
      //--- Exit deal(s) for position
      else if(deal.Entry()==DEAL_ENTRY_OUT || deal.Entry()==DEAL_ENTRY_OUT_BY)
        {
         pos_close_time      = deal.Time();
         pos_sum_cost       += deal.Volume() * deal.Price();
         pos_close_volume   += deal.Volume();
         pos_close_price     = pos_sum_cost / pos_close_volume;
         pos_sl              = HistoryDealGetDouble(deal.Ticket(), DEAL_SL);
         pos_tp              = HistoryDealGetDouble(deal.Ticket(), DEAL_TP);
         //---
         pos_close_comment  += deal.Comment() + " ";
         pos_deal_out       += IntegerToString(deal.Ticket()) + " ";
         pos_close_reason    = HistoryDealGetInteger(deal.Ticket(), DEAL_REASON);
        }
     }
//--- If the position is still open, it will not be displayed in the history.
   if(deals<2 || MathAbs(pos_open_volume-pos_close_volume)>0.00001)
     {
      Print(__FUNCTION__+" > Error: position with ticket #",position_ticket," is still open.");
      return(DBL_MIN);
     }
//---
   StringTrimLeft(pos_close_comment);
   StringTrimRight(pos_close_comment);
   StringTrimRight(pos_deal_out);
//---
   SymbolSelect(pos_symbol,true);
   int digits=(int)SymbolInfoInteger(pos_symbol,SYMBOL_DIGITS);
   double point=SymbolInfoDouble(pos_symbol,SYMBOL_POINT);
//---
   string acc_curr=AccountInfoString(ACCOUNT_CURRENCY);
//---
return pos_profit;
//---
  }
  
double get_profit_live(ulong tick)
{
   if(PositionSelectByTicket(tick))
   {
      return PositionGetDouble(POSITION_PROFIT);
   }
   return 0;
}

string ConvertTimeframeToString(ENUM_TIMEFRAMES timeframe)
{
   if(timeframe == PERIOD_CURRENT)
   {
      timeframe = (ENUM_TIMEFRAMES)Period();
   }

   switch(timeframe)
   {
      case PERIOD_M1:    return "M1";    // 1 minute
      case PERIOD_M2:    return "M2";    // 2 minutes
      case PERIOD_M3:    return "M3";    // 3 minutes
      case PERIOD_M4:    return "M4";    // 4 minutes
      case PERIOD_M5:    return "M5";    // 5 minutes
      case PERIOD_M6:    return "M6";    // 6 minutes
      case PERIOD_M10:   return "M10";   // 10 minutes
      case PERIOD_M12:   return "M12";   // 12 minutes
      case PERIOD_M15:   return "M15";   // 15 minutes
      case PERIOD_M20:   return "M20";   // 20 minutes
      case PERIOD_M30:   return "M30";   // 30 minutes
      case PERIOD_H1:    return "H1";    // 1 hour
      case PERIOD_H2:    return "H2";    // 2 hours
      case PERIOD_H3:    return "H3";    // 3 hours
      case PERIOD_H4:    return "H4";    // 4 hours
      case PERIOD_H6:    return "H6";    // 6 hours
      case PERIOD_H8:    return "H8";    // 8 hours
      case PERIOD_H12:   return "H12";   // 12 hours
      case PERIOD_D1:    return "D1";    // 1 day
      case PERIOD_W1:    return "W1";    // 1 week
      case PERIOD_MN1:   return "MN1";   // 1 month
      default:           return "Unknown"; // No valid period found
   }
}