#region Using declarations
using System;
using System.ComponentModel;
using System.ComponentModel.DataAnnotations;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Gui.Tools;
using NinjaTrader.Data;
using NinjaTrader.NinjaScript;
using NinjaTrader.NinjaScript.Indicators;
using NinjaTrader.NinjaScript.DrawingTools;
using System.Windows.Media;
using SharpDX.Direct2D1;
using System.Text;
using System.Windows;
using SharpDX;
using Brush = System.Windows.Media.Brush;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Gui; // Add this for DashStyleHelper
#endregion

// This namespace holds Indicators in this folder and is required. Do not change it. 
namespace NinjaTrader.NinjaScript.Indicators
{
    public class MACDCrossOver : Indicator
    {
        private MACD macdIndicator;
        private Series<double> fastEma;
        private Series<double> macdAvg2;
        private Series<double> signal;
        private Series<double> macdValueSeries;
        private double previousMacd;
        private double previousSignal;

        [Browsable(false)]
        [XmlIgnore]
        public Series<double> Gap { get; set; }

        protected override void OnStateChange()
        {
            if (State == State.SetDefaults)
            {
                Description = @"Show the gap between signal and MACD at bar break";
                Name = "MACDCrossOver";
                Calculate = Calculate.OnBarClose;
                IsOverlay = true;
                FastEMA = 12;
                SlowEMA = 26;
                SignalSmoothing = 9;
                ShiftBar = true; // Default shift value
                PlaySound = false; // Default value for playing sound
                BarsRequiredToTrade = 20;
                SoundFilePathUp = @"C:\Windows\Media\Windows Hardware Insert.wav"; // Default sound file path
                SoundFilePathDown = @"C:\Windows\Media\Windows Hardware Remove.wav"; // Default sound file path
                MACDZeroLag = false; // Default to standard MACD calculation

                // Default MACD Colors, DashStyles, and Widths
                MACDUptrendColor = Brushes.Green;
                MACDDowntrendColor = Brushes.Red;
                MACDUptrendDashStyle = DashStyleHelper.Solid;
                MACDDowntrendDashStyle = DashStyleHelper.Solid;
                MACDUptrendWidth = 2;
                MACDDowntrendWidth = 2;

                // Default Lag Zero Colors, DashStyles, and Widths
                LagZeroUptrendColor = Brushes.DarkCyan;
                LagZeroDowntrendColor = Brushes.Crimson;
                LagZeroUptrendDashStyle = DashStyleHelper.Solid;
                LagZeroDowntrendDashStyle = DashStyleHelper.Dash;
                LagZeroUptrendWidth = 2;  
                LagZeroDowntrendWidth = 2; 
            }
            else if (State == State.Configure)
            {
                // Additional configuration if needed
            }
            else if (State == State.DataLoaded)
            {
                if (MACDZeroLag)
                {
                    // Initialize series for Zero Lag MACD calculation
                    fastEma = new Series<double>(this);
                    macdAvg2 = new Series<double>(this);
                    signal = new Series<double>(this);
                    macdValueSeries = new Series<double>(this);
                }
                else
                {
                    macdIndicator = MACD(Close, FastEMA, SlowEMA, SignalSmoothing);
                }
                Gap = new Series<double>(this);
            }
        }

        protected override void OnRender(ChartControl chartControl, ChartScale chartScale)
        {
            base.OnRender(chartControl, chartScale);
        }

        protected override void OnBarUpdate()
        {
            if (CurrentBar < BarsRequiredToTrade)
                return;

            double macdValue;
            double signalValue;

            if (MACDZeroLag)
            {
                // Zero Lag MACD Calculation
                if (CurrentBar == 0)
                {
                    fastEma[0] = Input[0];
                    macdAvg2[0] = Input[0];
                    macdValueSeries[0] = 0;
                    signal[0] = 0;
                }
                else
                {
                    fastEma[0] = ZeroLagEMA(Input, FastEMA)[0] - ZeroLagEMA(Input, SlowEMA)[0];
                    double macd = fastEma[0];

                    macdAvg2[0] = ZeroLagEMA(fastEma, SignalSmoothing)[0];
                    double macdAvg = macdAvg2[0];

                    macdValueSeries[0] = macd;
                    signal[0] = macdAvg;
                }

                macdValue = macdValueSeries[0];
                signalValue = signal[0];
            }
            else
            {
                // Standard MACD Calculation
                macdValue = macdIndicator[0];
                signalValue = macdIndicator.Avg[0];
            }

            // Calculate the bar duration in minutes
            double barDurationMinutes = (Time[1] - Time[0]).TotalMinutes;

            // Shift by 0.25 or 0 minutes depending on ShiftBar value
            DateTime shiftedTime = Time[0].AddMinutes(-barDurationMinutes * 0.25 * (ShiftBar ? 1 : 0));

            // Select appropriate colors and styles based on the MACDZeroLag toggle
            Brush uptrendColor = MACDZeroLag ? LagZeroUptrendColor : MACDUptrendColor;
            Brush downtrendColor = MACDZeroLag ? LagZeroDowntrendColor : MACDDowntrendColor;
            DashStyleHelper uptrendDashStyle = MACDZeroLag ? LagZeroUptrendDashStyle : MACDUptrendDashStyle;
            DashStyleHelper downtrendDashStyle = MACDZeroLag ? LagZeroDowntrendDashStyle : MACDDowntrendDashStyle;
            int uptrendWidth = MACDZeroLag ? LagZeroUptrendWidth : MACDUptrendWidth;
            int downtrendWidth = MACDZeroLag ? LagZeroDowntrendWidth : MACDDowntrendWidth;

            // Check for MACD crossover and display text on the chart
            if (previousMacd <= previousSignal && macdValue > signalValue)
            {
                Draw.VerticalLine(this, "MACDCrossUp" + CurrentBar, shiftedTime, uptrendColor, uptrendDashStyle, uptrendWidth);
                if (PlaySound)
                    PlaySound(SoundFilePathUp); // Play sound for MACD cross up
            }
            else if (previousMacd >= previousSignal && macdValue < signalValue)
            {
                Draw.VerticalLine(this, "MACDCrossDown" + CurrentBar, shiftedTime, downtrendColor, downtrendDashStyle, downtrendWidth);
                if (PlaySound)
                    PlaySound(SoundFilePathDown); // Play sound for MACD cross down
            }

            // Update the previous values
            previousMacd = macdValue;
            previousSignal = signalValue;
        }

        #region Properties

        // MACD Visual Properties
        [NinjaScriptProperty]
        [XmlIgnore] // Non-serializable
        [Display(Name = "MACD Uptrend Color", Order = 1, GroupName = "MACD Visual")]
        public Brush MACDUptrendColor { get; set; }

        [NinjaScriptProperty]
        [XmlIgnore] // Non-serializable
        [Display(Name = "MACD Uptrend Dash Style", Order = 2, GroupName = "MACD Visual")]
        public DashStyleHelper MACDUptrendDashStyle { get; set; }

        [NinjaScriptProperty]
        [XmlIgnore] // Non-serializable
        [Display(Name = "MACD Downtrend Color", Order = 3, GroupName = "MACD Visual")]
        public Brush MACDDowntrendColor { get; set; }

        [NinjaScriptProperty]
        [XmlIgnore] // Non-serializable
        [Display(Name = "MACD Downtrend Dash Style", Order = 4, GroupName = "MACD Visual")]
        public DashStyleHelper MACDDowntrendDashStyle { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "MACD Uptrend Width", Order = 5, GroupName = "MACD Visual")]
        public int MACDUptrendWidth { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "MACD Downtrend Width", Order = 6, GroupName = "MACD Visual")]
        public int MACDDowntrendWidth { get; set; }

        // Calculation Type Properties
        [NinjaScriptProperty]
        [Display(Name = "Fast EMA", Order = 1, GroupName = "Calculation Type")]
        public int FastEMA { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "Slow EMA", Order = 2, GroupName = "Calculation Type")]
        public int SlowEMA { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "Signal Smoothing", Order = 3, GroupName = "Calculation Type")]
        public int SignalSmoothing { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "Shift Bar", Order = 5, GroupName = "Calculation Type")]
        public bool ShiftBar { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "Play Sound", Order = 6, GroupName = "Calculation Type")]
        public bool PlaySound { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "Uptrend Sound File Location", Order = 7, GroupName = "Calculation Type")]
        public string SoundFilePathUp { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "Downtrend Sound File Location", Order = 8, GroupName = "Calculation Type")]
        public string SoundFilePathDown { get; set; }

        // Lag Zero Visual Properties
        [NinjaScriptProperty]
        [Display(Name = "Use Zero Lag MACD", Order = 1, GroupName = "Lag Zero Visual")]
        public bool MACDZeroLag { get; set; }

		[NinjaScriptProperty]
        [XmlIgnore] // Non-serializable
        [Display(Name = "Lag Zero Uptrend Color", Order = 2, GroupName = "Lag Zero Visual")]
        public Brush LagZeroUptrendColor { get; set; }

        [NinjaScriptProperty]
        [XmlIgnore] // Non-serializable
        [Display(Name = "Lag Zero Uptrend Dash Style", Order = 3, GroupName = "Lag Zero Visual")]
        public DashStyleHelper LagZeroUptrendDashStyle { get; set; }

        [NinjaScriptProperty]
        [XmlIgnore] // Non-serializable
        [Display(Name = "Lag Zero Downtrend Color", Order = 4, GroupName = "Lag Zero Visual")]
        public Brush LagZeroDowntrendColor { get; set; }

        [NinjaScriptProperty]
        [XmlIgnore] // Non-serializable
        [Display(Name = "Lag Zero Downtrend Dash Style", Order = 5, GroupName = "Lag Zero Visual")]
        public DashStyleHelper LagZeroDowntrendDashStyle { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "Lag Zero Uptrend Width", Order = 6, GroupName = "Lag Zero Visual")]
        public int LagZeroUptrendWidth { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "Lag Zero Downtrend Width", Order = 7, GroupName = "Lag Zero Visual")]
        public int LagZeroDowntrendWidth { get; set; }

        [Browsable(false)]
        [XmlIgnore]
        public int BarsRequiredToTrade { get; set; }

        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.

namespace NinjaTrader.NinjaScript.Indicators
{
	public partial class Indicator : NinjaTrader.Gui.NinjaScript.IndicatorRenderBase
	{
		private MACDCrossOver[] cacheMACDCrossOver;
		public MACDCrossOver MACDCrossOver(Brush mACDUptrendColor, DashStyleHelper mACDUptrendDashStyle, Brush mACDDowntrendColor, DashStyleHelper mACDDowntrendDashStyle, int mACDUptrendWidth, int mACDDowntrendWidth, int fastEMA, int slowEMA, int signalSmoothing, bool shiftBar, bool playSound, string soundFilePathUp, string soundFilePathDown, bool mACDZeroLag, Brush lagZeroUptrendColor, DashStyleHelper lagZeroUptrendDashStyle, Brush lagZeroDowntrendColor, DashStyleHelper lagZeroDowntrendDashStyle, int lagZeroUptrendWidth, int lagZeroDowntrendWidth)
		{
			return MACDCrossOver(Input, mACDUptrendColor, mACDUptrendDashStyle, mACDDowntrendColor, mACDDowntrendDashStyle, mACDUptrendWidth, mACDDowntrendWidth, fastEMA, slowEMA, signalSmoothing, shiftBar, playSound, soundFilePathUp, soundFilePathDown, mACDZeroLag, lagZeroUptrendColor, lagZeroUptrendDashStyle, lagZeroDowntrendColor, lagZeroDowntrendDashStyle, lagZeroUptrendWidth, lagZeroDowntrendWidth);
		}

		public MACDCrossOver MACDCrossOver(ISeries<double> input, Brush mACDUptrendColor, DashStyleHelper mACDUptrendDashStyle, Brush mACDDowntrendColor, DashStyleHelper mACDDowntrendDashStyle, int mACDUptrendWidth, int mACDDowntrendWidth, int fastEMA, int slowEMA, int signalSmoothing, bool shiftBar, bool playSound, string soundFilePathUp, string soundFilePathDown, bool mACDZeroLag, Brush lagZeroUptrendColor, DashStyleHelper lagZeroUptrendDashStyle, Brush lagZeroDowntrendColor, DashStyleHelper lagZeroDowntrendDashStyle, int lagZeroUptrendWidth, int lagZeroDowntrendWidth)
		{
			if (cacheMACDCrossOver != null)
				for (int idx = 0; idx < cacheMACDCrossOver.Length; idx++)
					if (cacheMACDCrossOver[idx] != null && cacheMACDCrossOver[idx].MACDUptrendColor == mACDUptrendColor && cacheMACDCrossOver[idx].MACDUptrendDashStyle == mACDUptrendDashStyle && cacheMACDCrossOver[idx].MACDDowntrendColor == mACDDowntrendColor && cacheMACDCrossOver[idx].MACDDowntrendDashStyle == mACDDowntrendDashStyle && cacheMACDCrossOver[idx].MACDUptrendWidth == mACDUptrendWidth && cacheMACDCrossOver[idx].MACDDowntrendWidth == mACDDowntrendWidth && cacheMACDCrossOver[idx].FastEMA == fastEMA && cacheMACDCrossOver[idx].SlowEMA == slowEMA && cacheMACDCrossOver[idx].SignalSmoothing == signalSmoothing && cacheMACDCrossOver[idx].ShiftBar == shiftBar && cacheMACDCrossOver[idx].PlaySound == playSound && cacheMACDCrossOver[idx].SoundFilePathUp == soundFilePathUp && cacheMACDCrossOver[idx].SoundFilePathDown == soundFilePathDown && cacheMACDCrossOver[idx].MACDZeroLag == mACDZeroLag && cacheMACDCrossOver[idx].LagZeroUptrendColor == lagZeroUptrendColor && cacheMACDCrossOver[idx].LagZeroUptrendDashStyle == lagZeroUptrendDashStyle && cacheMACDCrossOver[idx].LagZeroDowntrendColor == lagZeroDowntrendColor && cacheMACDCrossOver[idx].LagZeroDowntrendDashStyle == lagZeroDowntrendDashStyle && cacheMACDCrossOver[idx].LagZeroUptrendWidth == lagZeroUptrendWidth && cacheMACDCrossOver[idx].LagZeroDowntrendWidth == lagZeroDowntrendWidth && cacheMACDCrossOver[idx].EqualsInput(input))
						return cacheMACDCrossOver[idx];
			return CacheIndicator<MACDCrossOver>(new MACDCrossOver(){ MACDUptrendColor = mACDUptrendColor, MACDUptrendDashStyle = mACDUptrendDashStyle, MACDDowntrendColor = mACDDowntrendColor, MACDDowntrendDashStyle = mACDDowntrendDashStyle, MACDUptrendWidth = mACDUptrendWidth, MACDDowntrendWidth = mACDDowntrendWidth, FastEMA = fastEMA, SlowEMA = slowEMA, SignalSmoothing = signalSmoothing, ShiftBar = shiftBar, PlaySound = playSound, SoundFilePathUp = soundFilePathUp, SoundFilePathDown = soundFilePathDown, MACDZeroLag = mACDZeroLag, LagZeroUptrendColor = lagZeroUptrendColor, LagZeroUptrendDashStyle = lagZeroUptrendDashStyle, LagZeroDowntrendColor = lagZeroDowntrendColor, LagZeroDowntrendDashStyle = lagZeroDowntrendDashStyle, LagZeroUptrendWidth = lagZeroUptrendWidth, LagZeroDowntrendWidth = lagZeroDowntrendWidth }, input, ref cacheMACDCrossOver);
		}
	}
}

namespace NinjaTrader.NinjaScript.MarketAnalyzerColumns
{
	public partial class MarketAnalyzerColumn : MarketAnalyzerColumnBase
	{
		public Indicators.MACDCrossOver MACDCrossOver(Brush mACDUptrendColor, DashStyleHelper mACDUptrendDashStyle, Brush mACDDowntrendColor, DashStyleHelper mACDDowntrendDashStyle, int mACDUptrendWidth, int mACDDowntrendWidth, int fastEMA, int slowEMA, int signalSmoothing, bool shiftBar, bool playSound, string soundFilePathUp, string soundFilePathDown, bool mACDZeroLag, Brush lagZeroUptrendColor, DashStyleHelper lagZeroUptrendDashStyle, Brush lagZeroDowntrendColor, DashStyleHelper lagZeroDowntrendDashStyle, int lagZeroUptrendWidth, int lagZeroDowntrendWidth)
		{
			return indicator.MACDCrossOver(Input, mACDUptrendColor, mACDUptrendDashStyle, mACDDowntrendColor, mACDDowntrendDashStyle, mACDUptrendWidth, mACDDowntrendWidth, fastEMA, slowEMA, signalSmoothing, shiftBar, playSound, soundFilePathUp, soundFilePathDown, mACDZeroLag, lagZeroUptrendColor, lagZeroUptrendDashStyle, lagZeroDowntrendColor, lagZeroDowntrendDashStyle, lagZeroUptrendWidth, lagZeroDowntrendWidth);
		}

		public Indicators.MACDCrossOver MACDCrossOver(ISeries<double> input , Brush mACDUptrendColor, DashStyleHelper mACDUptrendDashStyle, Brush mACDDowntrendColor, DashStyleHelper mACDDowntrendDashStyle, int mACDUptrendWidth, int mACDDowntrendWidth, int fastEMA, int slowEMA, int signalSmoothing, bool shiftBar, bool playSound, string soundFilePathUp, string soundFilePathDown, bool mACDZeroLag, Brush lagZeroUptrendColor, DashStyleHelper lagZeroUptrendDashStyle, Brush lagZeroDowntrendColor, DashStyleHelper lagZeroDowntrendDashStyle, int lagZeroUptrendWidth, int lagZeroDowntrendWidth)
		{
			return indicator.MACDCrossOver(input, mACDUptrendColor, mACDUptrendDashStyle, mACDDowntrendColor, mACDDowntrendDashStyle, mACDUptrendWidth, mACDDowntrendWidth, fastEMA, slowEMA, signalSmoothing, shiftBar, playSound, soundFilePathUp, soundFilePathDown, mACDZeroLag, lagZeroUptrendColor, lagZeroUptrendDashStyle, lagZeroDowntrendColor, lagZeroDowntrendDashStyle, lagZeroUptrendWidth, lagZeroDowntrendWidth);
		}
	}
}

namespace NinjaTrader.NinjaScript.Strategies
{
	public partial class Strategy : NinjaTrader.Gui.NinjaScript.StrategyRenderBase
	{
		public Indicators.MACDCrossOver MACDCrossOver(Brush mACDUptrendColor, DashStyleHelper mACDUptrendDashStyle, Brush mACDDowntrendColor, DashStyleHelper mACDDowntrendDashStyle, int mACDUptrendWidth, int mACDDowntrendWidth, int fastEMA, int slowEMA, int signalSmoothing, bool shiftBar, bool playSound, string soundFilePathUp, string soundFilePathDown, bool mACDZeroLag, Brush lagZeroUptrendColor, DashStyleHelper lagZeroUptrendDashStyle, Brush lagZeroDowntrendColor, DashStyleHelper lagZeroDowntrendDashStyle, int lagZeroUptrendWidth, int lagZeroDowntrendWidth)
		{
			return indicator.MACDCrossOver(Input, mACDUptrendColor, mACDUptrendDashStyle, mACDDowntrendColor, mACDDowntrendDashStyle, mACDUptrendWidth, mACDDowntrendWidth, fastEMA, slowEMA, signalSmoothing, shiftBar, playSound, soundFilePathUp, soundFilePathDown, mACDZeroLag, lagZeroUptrendColor, lagZeroUptrendDashStyle, lagZeroDowntrendColor, lagZeroDowntrendDashStyle, lagZeroUptrendWidth, lagZeroDowntrendWidth);
		}

		public Indicators.MACDCrossOver MACDCrossOver(ISeries<double> input , Brush mACDUptrendColor, DashStyleHelper mACDUptrendDashStyle, Brush mACDDowntrendColor, DashStyleHelper mACDDowntrendDashStyle, int mACDUptrendWidth, int mACDDowntrendWidth, int fastEMA, int slowEMA, int signalSmoothing, bool shiftBar, bool playSound, string soundFilePathUp, string soundFilePathDown, bool mACDZeroLag, Brush lagZeroUptrendColor, DashStyleHelper lagZeroUptrendDashStyle, Brush lagZeroDowntrendColor, DashStyleHelper lagZeroDowntrendDashStyle, int lagZeroUptrendWidth, int lagZeroDowntrendWidth)
		{
			return indicator.MACDCrossOver(input, mACDUptrendColor, mACDUptrendDashStyle, mACDDowntrendColor, mACDDowntrendDashStyle, mACDUptrendWidth, mACDDowntrendWidth, fastEMA, slowEMA, signalSmoothing, shiftBar, playSound, soundFilePathUp, soundFilePathDown, mACDZeroLag, lagZeroUptrendColor, lagZeroUptrendDashStyle, lagZeroDowntrendColor, lagZeroDowntrendDashStyle, lagZeroUptrendWidth, lagZeroDowntrendWidth);
		}
	}
}

#endregion
